package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.DictDataVo;
import cn.sinozg.applet.common.core.model.I18nName;
import cn.sinozg.applet.common.enums.DictType;
import cn.sinozg.applet.common.holder.UserContextHolder;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
* 国际化工具类
* @Author: xyb
* @Description:
* @Date: 2023-05-04 下午 12:34
**/
public class I18nUtil {

    private I18nUtil(){
    }

    /**
     * 获取到对应国际化的名称，根据用户缓存的语言
     * @param i18nName 国际化
     * @return label名称
     */
    public static String getDictName (I18nName i18nName){
        return getDictName(i18nName, null);
    }

    /**
     * 获取缓存名称 根据用户数据
     * @param cb 缓存对象
     * @return label名称
     */
    public static String getDictName (DictDataVo cb){
        return getDictName(cb, null);
    }

    /**
     * 根据语言获取到 缓存的名称
     * @param cb 缓存对象
     * @param locale 语言
     * @return label名称
     */
    public static String getDictName (DictDataVo cb, Locale locale){
        I18nName i18nName = cb.getI18nLabel();
        return getDictName(i18nName, locale);
    }
    /**
     * 根据key获取国际化数据字典值
     * @param type 根类型
     * @param code code 支持多个 逗号分割
     * @return label名称
     */
    public static String getDictName(DictType type, String code){
        return getDictName(null, type, code);
    }

    /**
     * 获取转码
     * @param locale 语言
     * @param type 类型
     * @param code 数据code
     * @return label名称
     */
    public static String getDictName(Locale locale, DictType type, String code){
        List<I18nName> list = names(type, code);
        StringBuffer names = new StringBuffer();
        // 获取到国际化配置
        if (locale == null) {
            locale = UserContextHolder.getLocale();
        }
        String temp ;
        if (CollectionUtils.isNotEmpty(list)) {
            for (I18nName n : list) {
                if (StringUtils.isNotBlank(names)) {
                    names.append(BaseConstants.COMMA);
                }
                temp = getDictName(n, locale);
                if (StringUtils.isNotBlank(temp)) {
                    names.append(temp);
                }
            }
        }
        return names.toString();
    }
    /**
     * 获取转码
     * @param type 类型
     * @param code code
     * @param language 语言
     * @return label名称
     */
    public static String getDictName(DictType type, String code, String language){
        Locale locale = new Locale(language);
        return getDictName(locale, type, code);
    }

    /**
     * 根据语言 获取到对应国际化的名称
     * @param i18nName 国际化
     * @param locale 语言
     * @return label名称
     */
    public static String getDictName (I18nName i18nName, Locale locale){
        if (i18nName == null) {
            return null;
        }
        if (locale == null){
            locale = UserContextHolder.getLocale();
        }
        if (Locale.US.equals(locale)) {
            return i18nName.getEnName();
        }
        return i18nName.getName();
    }


    /**
     * 根据缓存类型 和值 得到国际化对象
     * @param type 类型
     * @param code 值
     * @return 对应的国际化数据
     */
    public static I18nName i18nName (DictType type, String code){
        List<I18nName> list = names(type, code);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }


    /**
     * code 与 国际化数据的映射
     * @param type 类型
     * @return code 与 国际化数据的映射
     */
    public static Map<String, I18nName> nameByType(DictType type){
        Map<String, I18nName> map = new HashMap<>(64);
        List<DictDataVo> list = DictUtil.getDictCache(type);
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(c -> map.put(c.getDictValue(), c.getI18nLabel()));
        }
        return map;
    }
    /**
     * 根据缓存类型 和值 得到国际化对象
     * @param type 类型
     * @param code 值
     * @return 对应的国际化数据集合
     */
    private static List<I18nName> names (DictType type, String code){
        if (StringUtils.isBlank(code) || type == null) {
            return null;
        }
        String[] codes = StringUtils.split(code, BaseConstants.COMMA);
        Map<String, I18nName> map = nameByType(type);
        if (ArrayUtils.isEmpty(codes) || MapUtils.isEmpty(map)) {
            return null;
        }
        List<I18nName> names = new ArrayList<>();
        for (String c : codes) {
            names.add(map.get(c));
        }
        return names;
    }

    /**
     * 获取语言
     * @return 当前语言
     */
    public static String locale (){
        Locale locale = UserContextHolder.getLocale();
        return locale.toString();
    }
}
