package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.exception.CavException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;

import jakarta.servlet.http.HttpServletResponse;
import java.io.BufferedOutputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.function.Consumer;

/**
 * 文件下载 工具类
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-05 17:10
 */
public class DowUtil {

    private static final Logger log = LoggerFactory.getLogger(DowUtil.class);

    private DowUtil() {
    }
    /**
     * 下载指定资源
     * @param response HttpServletResponse
     * @param fileName 文件名
     * @param bs 输出的字节
     * @param consumer 执行方法
     */
    public static void download(HttpServletResponse response, String fileName, byte[] bs, Consumer<BufferedOutputStream> consumer)  {
        setResponse(response, fileName, bs);
        try (OutputStream os = response.getOutputStream();
             BufferedOutputStream bos = new BufferedOutputStream(os)) {
            if (bs != null) {
                bos.write(bs);
            }
            if (consumer != null) {
                consumer.accept(bos);
            }
        } catch (Exception e) {
            log.error("下载资源错误。", e);
            throw new CavException("BIZ000100023");
        }
    }

    /**
     * 设置 response
     * @param response response
     * @param fileName 文件名
     * @param bs 要输出的流
     */
    public static void setResponse (HttpServletResponse response, String fileName, byte[] bs){
        String charset = StandardCharsets.UTF_8.name();
        response.reset();
        response.setCharacterEncoding(charset);
        response.setHeader(HttpHeaders.CONTENT_ENCODING, charset);
        response.setContentType(FileUtil.fileType(fileName));
        if (bs != null) {
            response.setIntHeader(HttpHeaders.CONTENT_LENGTH, bs.length);
        }
        try {
            response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment;filename=" + URLEncoder.encode(fileName, charset));
        } catch (Exception ignored) {
        }
    }
}
