package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.exception.CavException;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.util.zip.CRC32;
import java.util.zip.CheckedOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * zip 工具类
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-06-21 11:32
 */
public class ZipUtil {

    private static final int BUFFER = 1024;

    private static final Logger log = LoggerFactory.getLogger(ZipUtil.class);
    private ZipUtil(){
    }
    /**
     * 将指定路径下的所有文件打包zip导出
     * @param response response
     * @param sourceFilePath 需要打包的文件夹路径
     * @param postfix 下载时的文件后缀 .zip/.rar
     */
    public static void exportZip(HttpServletResponse response, String sourceFilePath, String postfix) {
        // 默认文件名以时间戳作为前缀
        String downloadName = System.currentTimeMillis() + postfix;
        try (ByteArrayOutputStream byteOs = createZip(sourceFilePath)) {
            // 接收压缩包字节
            byte[] data = byteOs.toByteArray();
            DowUtil.download(response, downloadName, data, null);
        } catch (Exception e) {
            log.error("导出zip文件错误，", e);
            throw new CavException("导出zip文件错误！");
        }
    }


    /**
     * 压缩文件夹
     * @param fromPath 目录
     * @param toPath 目录
     */
    public static void compress(String fromPath, String toPath) {
        File fromFile = new File(fromPath);
        File toFile = new File(toPath);
        if (!fromFile.exists()) {
            throw new RuntimeException(fromPath + "不存在！");
        }
        try (FileOutputStream outputStream = new FileOutputStream(toFile);
             CheckedOutputStream checkedOutputStream = new CheckedOutputStream(outputStream, new CRC32());
             ZipOutputStream zipOutputStream = new ZipOutputStream(checkedOutputStream)) {
            compress(fromFile, zipOutputStream, StringUtils.EMPTY, true);
        } catch (Exception e) {
            log.error("压缩文件错误，", e);
            throw new CavException("压缩文件错误！");
        }
    }

    /**
     * 创建zip文件
     * @param sourceFilePath 源文件目录
     * @return 输出流
     */
    private static ByteArrayOutputStream createZip(String sourceFilePath) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try (CheckedOutputStream checkedOutputStream = new CheckedOutputStream(outputStream, new CRC32());
             ZipOutputStream zip = new ZipOutputStream(checkedOutputStream)
        ) {
            // 将目标文件打包成zip导出
            File file = new File(sourceFilePath);
            compress(file, zip, StringUtils.EMPTY, true);
        } catch (Exception e) {
            log.error("创建zip文件错误，", e);
            throw new CavException("创建zip文件错误！");
        }
        return outputStream;
    }

    /**
     * 压缩文件
     * @param file 文件
     * @param zipOut zipOut
     * @param baseDir 目录
     * @param root 是否根目录
     * @throws IOException 异常
     */
    private static void compress(File file, ZipOutputStream zipOut, String baseDir, boolean root) throws IOException {
        if (file.isDirectory()) {
            compressDirectory(file, zipOut, baseDir, root);
        } else {
            compressFile(file, zipOut, baseDir);
        }
    }

    /**
     * 压缩文件
     * @param file 文件
     * @param zipOut zipOut
     * @param baseDir 目录
     * @throws IOException 异常
     */
    private static void compressFile(File file, ZipOutputStream zipOut, String baseDir) throws IOException {
        if (!file.exists()) {
            return;
        }
        try (BufferedInputStream bis = new BufferedInputStream(Files.newInputStream(file.toPath()))) {
            ZipEntry entry = new ZipEntry(baseDir + file.getName());
            zipOut.putNextEntry(entry);
            int count;
            byte[] data = new byte[BUFFER];
            while ((count = bis.read(data, 0, BUFFER)) != -1) {
                zipOut.write(data, 0, count);
            }
            zipOut.closeEntry();
            zipOut.flush();
        }
    }

    /**
     * 压缩文件夹
     * @param dir 文件
     * @param zipOut zipOut
     * @param baseDir 目录
     * @param root 是否为根目录
     * @throws IOException 异常
     */
    private static void compressDirectory(File dir, ZipOutputStream zipOut, String baseDir, boolean root) throws IOException {
        File[] files = dir.listFiles();
        if (files != null && ArrayUtils.isNotEmpty(files)) {
            for (File file : files) {
                String dirPath = root ? baseDir : baseDir + dir.getName() + File.separator;
                compress(file, zipOut, dirPath, false);
            }
        }
    }
}
