/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.holder;

import cn.sinozg.applet.common.constant.BaseConstants;
import com.alibaba.ttl.TransmittableThreadLocal;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * 存储 traceId 链路 跟踪 id
 * @Author: xyb
 * @Description:
 * @Date: 2023-05-10 下午 01:28
 **/
public class MdcContextHolder {

    private static final Logger log = LoggerFactory.getLogger(MdcContextHolder.class);
    /** traceId */
    private static final ThreadLocal<Map<String, String>> MDC_HOLDER = new TransmittableThreadLocal<>() {
        @Override
        protected void beforeExecute() {
            final Map<String, String> map = get();
            map.forEach(MDC::put);
        }

        @Override
        protected void afterExecute() {
            MDC.clear();
        }

        @Override
        protected Map<String, String> initialValue() {
            return new HashMap<>(16);
        }
    };

    /**
     * 设置mdc信息
     * @param traceId traceId
     */
    public static void put(String traceId){
        if (StringUtils.isBlank(traceId)) {
            traceId = UUID.randomUUID().toString().replace(BaseConstants.MIDDLE_LINE, StringUtils.EMPTY);
        }
        MDC.put(BaseConstants.TRACE_ID, traceId);
        MDC_HOLDER.get().put(BaseConstants.TRACE_ID, traceId);
    }

    /**
     * 获取到链路 id
     * @return 链路 id
     */
    public static String get(){
        return MDC.get(BaseConstants.TRACE_ID);
    }

    /**
     * 清除上下文数据
     */
    public static void clear() {
        MDC.clear();
        MDC_HOLDER.get().clear();
        MDC_HOLDER.remove();
    }
}