/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.exception.CavException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.HibernateValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ClassUtils;
import org.springframework.validation.BindingResult;
import org.springframework.validation.DataBinder;
import org.springframework.validation.ObjectError;
import org.springframework.validation.SmartValidator;
import org.springframework.validation.beanvalidation.SpringValidatorAdapter;

import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import jakarta.validation.Validation;
import jakarta.validation.Validator;
import jakarta.validation.ValidatorFactory;
import java.util.List;
import java.util.Set;

/**
 * javaBean 参数手动校验
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-26 17:50
 */
public class VerifyUtil {
    private static Validator validator;
    private static SmartValidator validatorAdapter;

    /** 日志 */
    private static final Logger LOG = LoggerFactory.getLogger(VerifyUtil.class);

    static {
        // 快速返回模式
        try (ValidatorFactory factory = Validation.byProvider(HibernateValidator.class)
                .configure()
                .failFast(true)
                .buildValidatorFactory()){
            validator = factory.getValidator();
        } catch (Exception e) {
            LOG.error("参数校验初始化失败！", e);
        }
    }
    private VerifyUtil(){
    }

    public static Validator getValidator() {
        return validator;
    }

    private static SmartValidator getValidatorAdapter(Validator validator) {
        if (validatorAdapter == null) {
            validatorAdapter = new SpringValidatorAdapter(validator);
        }
        return validatorAdapter;
    }

    /**
     * 校验参数，用于普通参数校验
     * @param t 参数
     */
    public static <T> void validate(T t) {
        Set<ConstraintViolation<Object>> sets = validator.validate(t);
        if (!sets.isEmpty()) {
            throw new ConstraintViolationException(sets);
        }
    }

    /**
     * 校验对象
     * @param object 对象
     * @param groups 组
     * @param <T> 类型
     */
    public static <T> void validate(T object, Class<?>... groups) {
        Set<ConstraintViolation<T>> sets = validator.validate(object, groups);
        if (!sets.isEmpty()) {
            throw new ConstraintViolationException(sets);
        }
    }

    /**
     * 校验对象 使用与 Spring 集成的校验方式。
     * @param object 待校验对象
     * @param groups 待校验的组
     * @param <T> 类型
     */
    public static <T> void validateBySpring(T object, Class<?>... groups) {
        DataBinder dataBinder = getBinder(object);
        if (groups != null) {
            dataBinder.validate((Object[]) groups);
        }
        if (dataBinder.getBindingResult().hasErrors()) {
            throw new CavException("BIZ000100014", errorInfo(dataBinder.getBindingResult()));
        }
    }

    /**
     * 获取参数错误信息
     * @param result 错误信息
     * @return 转化
     */
    public static String errorInfo (BindingResult result){
        List<ObjectError> errors = result.getAllErrors();
        StringBuilder msg = new StringBuilder();
        if (CollectionUtils.isNotEmpty(errors)) {
            for (ObjectError oe : errors) {
                if (StringUtils.isNotBlank(msg)) {
                    msg.append(BaseConstants.COMMA);
                }
                msg.append(oe.getDefaultMessage());
            }
        }
        return msg.toString();
    }

    /**
     * 获取绑定
     * @param object 参数
     * @return 绑定
     * @param <T> 类型
     */
    private static <T> DataBinder getBinder(T object) {
        DataBinder dataBinder = new DataBinder(object, ClassUtils.getShortName(object.getClass()));
        dataBinder.setValidator(getValidatorAdapter(validator));
        return dataBinder;
    }
}
