/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.holder;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.LoginUserVo;
import cn.sinozg.applet.common.core.model.UserThreadInfo;
import cn.sinozg.applet.common.service.FrameworkAuthService;
import cn.sinozg.applet.common.utils.SpringUtil;
import cn.sinozg.applet.common.utils.UserUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import com.alibaba.ttl.TransmittableThreadLocal;
import jakarta.servlet.http.HttpServletRequest;
import java.util.Locale;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;

/**
 * 存储 用户上下文信息，用于多线程模式下获取到用户信息
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-26 下午 01:28
 **/
public class UserContextHolder {

    private static final Logger log = LoggerFactory.getLogger(UserContextHolder.class);
    /** 国际化缓存 */
    private static final ThreadLocal<Locale> LOCALE_HOLDER = new TransmittableThreadLocal<>();

    private static final ThreadLocal<UserThreadInfo> CONTEXT_OLDER = new TransmittableThreadLocal<>();

    private static final FrameworkAuthService AUTH_SERVICE = SpringUtil.getBean(FrameworkAuthService.class);

    /**
     * 设置用户信息
     * @param request HttpServletRequest
     */
    public static void setInfo(HttpServletRequest request){
        Locale locale = localByRequest(request);
        LoginUserVo user = UserUtil.user();
        if (user != null) {
            String token = AUTH_SERVICE.token();
            setInfo(user.getId(), token, user.getTenantId());
        }
        LOCALE_HOLDER.set(locale);
    }


    /**
     * 非 web环境下设置
     * @param uid 用户id
     * @param token token
     * @param tenantId 租户id
     */
    public static void setInfo(String uid, String token, String tenantId){
        UserThreadInfo info = new UserThreadInfo();
        info.setUid(uid);
        info.setToken(token);
        info.setTenantId(tenantId);
        CONTEXT_OLDER.set(info);
    }

    /**
     * 设置租户
     * @param tenantId 租户id
     */
    public static void setTenantId(String tenantId){
        UserThreadInfo user =  CONTEXT_OLDER.get();
        if (user == null) {
            user = new UserThreadInfo();
            user.setTenantId(tenantId);
        }
        CONTEXT_OLDER.set(user);
    }

    /**
     * 获取租户
     * @return 租户
     */
    public static String getTenantId(){
        UserThreadInfo user =  CONTEXT_OLDER.get();
        if (user != null) {
            return user.getTenantId();
        }
        return null;
    }

    /**
     * 获取语言
     * @return 语言
     */
    public static Locale getLocale (){
        return getLocale(WebUtil.request());
    }

    /**
     * 取得语言
     * @param request request
     * @return 语言
     */
    public static Locale getLocale(HttpServletRequest request) {
        Locale locale = LOCALE_HOLDER.get();
        if (locale == null) {
            setInfo(request);
            locale = LOCALE_HOLDER.get();
        }
        return locale;
    }

    /**
     * 获取到用户信息
     * @return 用户信息
     */
    public static UserThreadInfo getInfo (){
        return CONTEXT_OLDER.get();
    }

    /**
     * 清除上下文数据
     */
    public static void clear() {
        CONTEXT_OLDER.remove();
        LOCALE_HOLDER.remove();
    }

    /**
     * en-US,fr-CA
     * en-US,en;q=0.9
     * 从请求头里面获取到语言
     * @param request request
     * @return 语言
     */
    private static Locale localByRequest(HttpServletRequest request) {
        Locale locale = Locale.CHINA;
        if (request != null) {
            String headerValue = request.getHeader(HttpHeaders.ACCEPT_LANGUAGE);
            if (StringUtils.isNotBlank(headerValue)) {
                String acceptLanguage = headerValue.split(BaseConstants.COMMA)[0];
                locale = locale(acceptLanguage);
            }
        }
        return locale;
    }

    /**
     * 设置语言
     * @param language 语言
     */
    public static void setLocale(String language) {
        LOCALE_HOLDER.set(locale(language));
    }

    /**
     * 解析语言
     * @param acceptLanguage 语言
     * @return 语言
     */
    private static Locale locale (String acceptLanguage){
        String[] languages = null;
        if (StringUtils.contains(acceptLanguage, BaseConstants.MIDDLE_LINE)) {
            languages = acceptLanguage.split(BaseConstants.MIDDLE_LINE);
        }
        if (StringUtils.contains(acceptLanguage, StringUtils.SPACE)) {
            languages = acceptLanguage.split(BaseConstants.UNDERLINE);
        }
        // 语言长度
        int languageLength = 2;
        if (languages != null && languages.length == languageLength) {
            return new Locale(languages[0], languages[1]);
        }
        return Locale.CHINA;
    }
}