/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.DictDataVo;
import cn.sinozg.applet.common.core.model.I18nName;
import cn.sinozg.applet.common.enums.DictType;
import cn.sinozg.applet.common.holder.UserContextHolder;
import jakarta.servlet.http.HttpServletRequest;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.MessageSource;
import org.springframework.web.servlet.LocaleResolver;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;

/**
 * 国际化工具类<br/>
 * 支持中文：zh-CN,zh 法语:fr_FR 英文:en-US 日语:ja_JP 西班牙语:es_ES
 * @Description
 * @Copyright Copyright (c) 2024
 * @author xyb
 * @since 2023-05-04 17:26:33
*/ 
public class I18nUtil extends I18nBaseUtil {

    /** 消息 */
    private static final MessageSource MS = SpringUtil.getBean(MessageSource.class);
    /** 国际化 */
    private static final LocaleResolver LR = SpringUtil.getBean(BaseConstants.LOCALE_RESOLVER);

    private I18nUtil(){
    }

    /**
     * 获取到对应国际化的名称，根据用户缓存的语言
     * @param i18nName 国际化
     * @return label名称
     */
    public static String getDictName (I18nName i18nName){
        return getDictName(i18nName, null);
    }

    /**
     * 获取缓存名称 根据用户数据
     * @param cb 缓存对象
     * @return label名称
     */
    public static String getDictName (DictDataVo cb){
        return getDictName(cb, null);
    }

    /**
     * 根据语言获取到 缓存的名称
     * @param cb 缓存对象
     * @param locale 语言
     * @return label名称
     */
    public static String getDictName (DictDataVo cb, Locale locale){
        I18nName i18nName = cb.getI18nLabel();
        return getDictName(i18nName, locale);
    }
    /**
     * 根据key获取国际化数据字典值
     * @param type 根类型
     * @param code code 支持多个 逗号分割
     * @return label名称
     */
    public static String getDictName(DictType type, String code){
        return getDictName(null, type, code);
    }

    /**
     * 获取转码
     * @param locale 语言
     * @param type 类型
     * @param code 数据code
     * @return label名称
     */
    public static String getDictName(Locale locale, DictType type, String code){
        List<I18nName> list = names(type, code);
        StringBuffer names = new StringBuffer();
        // 获取到国际化配置
        if (locale == null) {
            locale = UserContextHolder.getLocale();
        }
        String temp ;
        if (CollectionUtils.isNotEmpty(list)) {
            for (I18nName n : list) {
                if (StringUtils.isNotBlank(names)) {
                    names.append(BaseConstants.COMMA);
                }
                temp = getDictName(n, locale);
                if (StringUtils.isNotBlank(temp)) {
                    names.append(temp);
                }
            }
        }
        return names.toString();
    }
    /**
     * 获取转码
     * @param type 类型
     * @param code code
     * @param language 语言
     * @return label名称
     */
    public static String getDictName(DictType type, String code, String language){
        Locale locale = new Locale(language);
        return getDictName(locale, type, code);
    }

    /**
     * 根据语言 获取到对应国际化的名称
     * @param i18nName 国际化
     * @param locale 语言
     * @return label名称
     */
    public static String getDictName (I18nName i18nName, Locale locale){
        if (i18nName == null) {
            return null;
        }
        if (locale == null){
            locale = UserContextHolder.getLocale();
        }
        return localeFunction(locale).apply(i18nName);
    }


    /**
     * 根据缓存类型 和值 得到国际化对象
     * @param type 类型
     * @param code 值
     * @return 对应的国际化数据
     */
    public static I18nName i18nName (DictType type, String code){
        List<I18nName> list = names(type, code);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }


    /**
     * code 与 国际化数据的映射
     * @param type 类型
     * @return code 与 国际化数据的映射
     */
    public static Map<String, I18nName> nameByType(DictType type){
        Map<String, I18nName> map = new HashMap<>(64);
        List<DictDataVo> list = DictUtil.getDictCache(type);
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(c -> map.put(c.getDictValue(), c.getI18nLabel()));
        }
        return map;
    }
    /**
     * 根据缓存类型 和值 得到国际化对象
     * @param type 类型
     * @param code 值
     * @return 对应的国际化数据集合
     */
    private static List<I18nName> names (DictType type, String code){
        if (StringUtils.isBlank(code) || type == null) {
            return null;
        }
        String[] codes = StringUtils.split(code, BaseConstants.COMMA);
        Map<String, I18nName> map = nameByType(type);
        if (ArrayUtils.isEmpty(codes) || MapUtils.isEmpty(map)) {
            return null;
        }
        List<I18nName> names = new ArrayList<>();
        for (String c : codes) {
            names.add(map.get(c));
        }
        return names;
    }

    /**
     * 获取语言
     * @return 当前语言
     */
    public static String locale (){
        Locale locale = UserContextHolder.getLocale();
        return locale.toString();
    }

    /**
     * 根据语种获取到 名称函数
     * @param locale 语种
     * @return 函数
     */
    public static Function<I18nName, String> localeFunction(Locale locale) {
        if (!LOCALE_MAP.containsKey(locale)) {
            locale = Locale.CHINA;
        }
        return LOCALE_MAP.get(locale);
    }

    /**
     * 通过 code 获取到国际化配置
     * @param propertyCode code
     * @param params 参数
     * @return 国际化信息
     */
    public static String propertyName (String propertyCode, Object ... params) {
        return propertyName(null, propertyCode, params);
    }
    /**
     * 通过 code 获取到国际化配置
     * @param request 请求
     * @param propertyCode code
     * @param params 参数
     * @return 国际化信息
     */
    public static String propertyName (HttpServletRequest request, String propertyCode, Object ... params) {
        Locale locale;
        if (LR == null) {
            locale = Locale.getDefault();
        } else {
            if (request == null) {
                request = WebUtil.request();
            }
            locale = LR.resolveLocale(request);
        }
        return MS.getMessage(propertyCode, params, locale);
    }

    /**
     * 项目是否开启国际化
     * @return 是狗国际化
     */
    public static boolean isI18n(){
        return SystemConfig.APP.getLogin().isI18n();
    }
}
