/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;


import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.base.BasePageResponse;
import cn.sinozg.applet.common.core.base.BaseRequest;
import cn.sinozg.applet.common.core.base.BaseResponse;
import cn.sinozg.applet.common.core.base.ErrorInfo;
import cn.sinozg.applet.common.core.base.PagingRequest;
import cn.sinozg.applet.common.core.base.PagingResponse;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.service.CipherService;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.function.BiFunction;

/**
 * 请求和返回封装
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:04
 **/
public class MsgUtil {

    private static final Logger log = LoggerFactory.getLogger(MsgUtil.class);
    /** 加密实现 */
    private static final CipherService CIPHER_SERVICE = SpringUtil.getBean(CipherService.class);
    private MsgUtil(){
    }

    /**
     * 获取到请求参数 做统一返回
     * @param request 请求对象
     * @return 请求参数
     * @param <T> 请求参数对象
     */
    public static <T> T params (BaseRequest<T> request){
        T t = request.getParams();
        if (t == null) {
            throw new CavException("BIZ000100000");
        }
        return t;
    }

    /**
     * 返回具体的参数
     * @param params 返回参数
     * @return 统一的返回结果
     * @param <T> 返参类型
     */
    public static <T> BaseResponse<T> ok(T params){
        BaseResponse<T> response = new BaseResponse<>();
        response.setData(params);
        return response;
    }

    /**
     * 返回默认的参数 即不带任何信息返回
     * @return 统一的返回结果
     */
    public static BaseResponse<String> ok(){
        return new BaseResponse<>();
    }

    /**
     * 转为分页结果
     * @param info 分页请求
     * @param r 查询参数
     * @param function 函数
     * @return 结果集
     * @param <T> 出参类型
     * @param <R> 入参类型
     */
    public static <T, R> BasePageResponse<List<T>> page(PagingRequest info, R r, BiFunction<Page<T>, R, IPage<T>> function){
        BasePageResponse<List<T>> response = new BasePageResponse<>();
        if (info == null) {
            throw new CavException("BIZ000100001");
        }
        long pageSize = info.getPageSize();
        Page<T> p = new Page<>(info.getPageNum(), pageSize);
        IPage<T> pageData = function.apply(p, r);
        PagingResponse page = new PagingResponse();
        page.setPageNum(pageData.getCurrent());
        page.setPages(pageData.getPages());
        page.setPageSize(pageSize);
        page.setTotalNum(pageData.getTotal());
        response.setPage(page);
        response.setData(pageData.getRecords());
        return response;
    }
    /**
     * 打印错误信息 带http错误码
     *  <p>直接writer json到页面
     * @param response response
     * @param request request
     * @param e 异常信息
     * @param status HTTP状态码
     * @param code 业务错误码
     * @param params 业务消息参数
     */
    public static void httpError(HttpServletResponse response, HttpServletRequest request, Exception e, HttpStatus status, String code, Object ... params) {
        BaseResponse<String> result = error(request, e, code, params);
        writeJson(response, request, status, result);
    }

    /**
     * 写json
     * @param response response
     * @param request request
     * @param status HTTP状态码
     * @param data 参数
     */
    public static void writeJson (HttpServletResponse response, HttpServletRequest request, HttpStatus status, Object data){
        response.setCharacterEncoding(BaseConstants.UTF8);
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        if (status != null) {
            response.setStatus(status.value());
        }
        try (PrintWriter writer = response.getWriter()){
            Object o = CIPHER_SERVICE.encryptJson(request, data);
            writer.print(JsonUtil.toJson(o));
        } catch (IOException ex) {
            log.error("response error", ex);
        }
    }


    /**
     * 获取到错误信息 并且打印错误日志
     * @param request http request
     * @param e 异常
     * @param code 业务code
     * @param params 错误参数
     * @return 统一的返回参数
     */
    public static BaseResponse<String> error (HttpServletRequest request, Exception e, String code, Object ... params) {
        return error(request, e, false, code, params);
    }

    /**
     * 获取到错误信息 并且打印错误日志
     * @param request http request
     * @param e 异常
     * @param debug debug 模式
     * @param code 业务code
     * @param params 错误参数
     * @return 统一的返回参数
     */
    public static BaseResponse<String> error (HttpServletRequest request, Exception e, boolean debug, String code, Object ... params) {
        ImmutablePair<String, String> pair = errorMsg(request, e, code, params);
        BaseResponse<String> response = new BaseResponse<>();
        response.setBizStatus(BaseConstants.FAIL);
        ErrorInfo r = new ErrorInfo();
        r.setCode(pair.getKey());
        r.setMessage(pair.getValue());
        r.setUrl(request.getRequestURI());
        log(pair.getValue(), e, request);
        if (debug) {
            r.setMessage(e.getMessage());
        }
        response.setErrorInfo(r);
        return response;
    }

    /**
     * 获取自定义异常的消息
     * @param request 请求
     * @param e 异常
     * @return 消息
     */
    public static String error (HttpServletRequest request, CavException e){
        if (request == null) {
            request = WebUtil.request();
        }
        return errorMsg(request, e, e.getCode(), e.getParameters()).getValue();
    }

    /**
     * 获取到消息
     * @param request 参数
     * @param e 异常
     * @param code 编码
     * @param params 参数
     * @return 获取到消息
     */
    private static ImmutablePair<String, String> errorMsg (HttpServletRequest request, Exception e, String code, Object ... params) {
        Throwable t = e;
        while (t != null && !(t instanceof CavException)) {
            t = t.getCause();
        }
        String c = code;
        Object[] ps = params;
        if (t != null) {
            CavException ce = (CavException) t;
            c = ce.getCode();
            ps = ce.getParameters();
        }
        String propertyName = I18nUtil.propertyName(request, c, ps);
        return ImmutablePair.of(c, propertyName);
    }

    /**
     * 打印错误消息
     * @param localMessage 国际化消息
     * @param ex 异常
     * @param request http request
     */
    private static void log(String localMessage, Exception ex, HttpServletRequest request) {
        if (ex == null) {
            return;
        }
        log.error("************************异常开始*******************************");
        log.error("请求地址：{}", request.getRequestURL());
        Enumeration<String> es = request.getParameterNames();
        log.error("请求参数");
        String key;
        while (es.hasMoreElements()) {
            key = es.nextElement();
            log.error("{}-------{}", key, request.getParameter(key));
        }
        if (ex instanceof CavException e) {
            Object[] os = Arrays.stream(e.getParameters()).filter(Objects::nonNull).toArray();
            log.error("Error : {} {} {}", e.getCode(), localMessage, ArrayUtils.toString(os));
        }
        log.error("Error DETAIL : ", ex);
        // 递归打印Cause
        printCause(ex.getCause(), ex.getStackTrace());
        log.error("************************异常结束*******************************");
    }


    /**
     * 递归CauseBy
     *
     * @param cause cause
     * @param pste 栈堆
     */
    public static void printCause(Throwable cause, StackTraceElement[] pste) {
        if (cause != null) {
            printCause(cause.getStackTrace());
            printCause(cause.getCause(), null);
        } else {
            printCause(pste);
        }
    }

    /**
     * 打印栈堆信息
     * @param ste 栈堆
     */
    private static void printCause (StackTraceElement[] ste){
        if (ArrayUtils.isNotEmpty(ste)) {
            log.error("Cause by: {}", ste[0].toString());
        }
    }
}
