/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.properties.SignValue;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.StandardReflectionParameterNameDiscoverer;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.web.servlet.HandlerExceptionResolver;

/**
 * JavaBean web 相关工具类
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-05 17:12
 */
public class PojoWebUtil {
    private PojoWebUtil(){
    }

    /**
     * 批量分批次新增 或者修改数据
     * 先把数据按照 500 分成多批次
     * 再从数据库根据联合主键查询数据，转成联合主键为key 对象为value的map
     * 再把集合遍历 分为更新的集合 和 修改的集合
     * 再分别批次更新或者修改
     * @param list 原始数据
     * @param function 查询数据库的方法 通过list 参数返回一个map对象
     * @param isUpdFun 判断是否为新增的数据 如果是修改需要再函数里设置主键
     * @param service service对象
     * @param <T> 实体类型
     * @param <S> service
     */
    public static <T, S extends ServiceImpl<?, T>> Map<String, T> batchAddOrUpdate (List<T> list, Function<List<T>, Map<String, T>> function, BiFunction<T, Map<String, T>, Boolean> isUpdFun, S service){
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        Map<String, T> map = new HashMap<>(64);
        List<Map<String, T>> partitions = PojoUtil.batchList(list, function);
        if (CollectionUtils.isNotEmpty(partitions)) {
            partitions.forEach(map::putAll);
        }
        List<T> insertList = new ArrayList<>();
        List<T> updateList = new ArrayList<>();
        list.forEach(l -> {
            boolean isUpdate = isUpdFun.apply(l, map);
            if (isUpdate) {
                updateList.add(l);
            } else {
                insertList.add(l);
            }
        });
        if (CollectionUtils.isNotEmpty(insertList)) {
            service.saveBatch(insertList);
        }
        if (CollectionUtils.isNotEmpty(updateList)) {
            service.updateBatchById(updateList);
        }
        return map;
    }


    /**
     * 从 aop里面获取到 注解
     * @param joinPoint joinPoint
     * @param clazz 注解类型
     * @return 注解对象
     * @param <T> 注解对象类型
     */
    public static <T extends Annotation> T getAnnotation(JoinPoint joinPoint, Class<T> clazz){
        T t = joinPoint.getTarget().getClass().getAnnotation(clazz);
        if (t == null) {
            Signature signature = joinPoint.getSignature();
            MethodSignature methodSignature = (MethodSignature) signature;
            Method method = methodSignature.getMethod();
            t = method.getAnnotation(clazz);
        }
        return t;
    }

    /**
     * 获取到 切面方法的唯一值 packageName + separator + 方法名称
     * @param jp 切面
     * @param separator 分隔符
     * @return key
     */
    public static String methodKey (JoinPoint jp, String separator){
        String className = jp.getTarget().getClass().getName();
        String name = RegExUtils.replacePattern(className, "\\.", StringUtils.EMPTY);
        return name + separator + jp.getSignature().getName();
    }

    /**
     * 解析 获取到注解上的值，支持spEL表达式
     * @param jp 注解
     * @param key key
     * @return 值
     */
    public static String getElValue (JoinPoint jp, String key) {
        String regex = "^#.*.$";
        if (key.matches(regex)) {
            Method method = ((MethodSignature) jp.getSignature()).getMethod();
            // 获取被拦截方法参数名列表(使用Spring支持类库)
            StandardReflectionParameterNameDiscoverer u = new StandardReflectionParameterNameDiscoverer();
            String[] params = u.getParameterNames(method);
            // 使用spEL进行key的解析
            ExpressionParser parser = new SpelExpressionParser();
            // spEL上下文
            StandardEvaluationContext context = new StandardEvaluationContext();
            // 把方法参数放入spEL上下文中
            if (params != null) {
                for (int i = 0; i < params.length; i++) {
                    context.setVariable(params[i], jp.getArgs()[i]);
                }
            }
            return parser.parseExpression(key).getValue(context, String.class);
        }
        return key;
    }

    /**
     * 处理异常
     * @param request request
     * @param response response
     * @param e 异常
     */
    public static void resolveException (HttpServletRequest request, HttpServletResponse response, Exception e) {
        HandlerExceptionResolver resolver = SpringUtil.getBean(BaseConstants.EXCEPTION_RESOLVER);
        if (resolver != null) {
            resolver.resolveException(request, response, null, e);
        }
    }

    /**
     * 判断接口是否需要加密
     * @param sign 加密参数
     * @param request 请求
     * @return 是否需要加密
     */
    public static boolean nonEncrypt (SignValue sign, HttpServletRequest request) {
        if (sign == null) {
            sign = SystemConfig.APP.getSign();
        }
        boolean isEncrypt = sign.isCryptoEnable();
        if (isEncrypt) {
            String encryptHeader = request.getHeader(HeaderConstants.X_NO_ENCRYPT);
            return StringUtils.isBlank(encryptHeader) || !StringUtils.equals(encryptHeader, "false");
        }
        return false;
    }
}
