/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.support.SFunction;
import com.baomidou.mybatisplus.extension.service.IService;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.RandomAccess;
import java.util.UUID;
import java.util.function.Function;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-05 20:07
 */
public class StringUtil {

    private StringUtil(){
    }
    /**
     * 根据图片id转为直接显示的地址
     * @param picId 图片id
     * @return 显示地址
     */
    public static String showUrl (String picId){
        if (StringUtils.isEmpty(picId)) {
            return null;
        }
        String host = SystemConfig.APP.getFontUrl();
        if (StringUtils.contains(picId, host)) {
            return picId;
        }
        return host + picId;
    }

    /**
     * 删除最后一个字符
     * @param input 参数
     */
    public static void delLast(StringBuilder input){
        if (!input.isEmpty()) {
            input.deleteCharAt(input.length() - 1);
        }
    }

    /**
     * 随机数 不要中间线
     * @return 随机数
     */
    public static String uuidNon (){
        return uuid().replace(BaseConstants.MIDDLE_LINE, StringUtils.EMPTY);
    }

    /**
     * 随机数
     * @return 随机数
     */
    public static String uuid (){
        return UUID.randomUUID().toString();
    }

    /**
     * 获取一个由字母和数据组成的随机数code，且判断是否已经存在<br/>
     * 比如订单号
     * @param len 随机数长度
     * @param prefix 前缀
     * @param fieldFun 对应的字段
     * @param service 数据库查询 对应的service
     * @return 新的编号
     * @param <T> 类型
     */
    public static <T> String randomBizCode(int len, String prefix, SFunction<T, String> fieldFun, IService<T> service){
        int batch = 10;
        RandomStringUtils secure = RandomStringUtils.secure();
        while (true) {
            List<String> codes = new ArrayList<>();
            for (int i = 0; i < batch; i++) {
                codes.add(prefix + secure.nextAlphanumeric(len));
            }
            List<T> list = service.list(new LambdaQueryWrapper<T>().in(fieldFun, codes));
            if (!(CollectionUtils.isNotEmpty(list) && list.size() == batch)) {
                List<String> dbCodes = PojoUtil.toList(list, fieldFun);
                for (String code : codes) {
                    if (!dbCodes.contains(code)) {
                        return code;
                    }
                }
            }
        }
    }

    /**
     * 获取一个由字母和数据组成的随机数code，且判断是否已经存在<br/>
     * 比如订单号
     * @param len 随机数长度
     * @param fieldFun 对应的字段
     * @param service 数据库查询 对应的service
     * @return 新的编号
     * @param <T> 类型
     */
    public static <T> String randomBizCode(int len, SFunction<T, String> fieldFun, IService<T> service){
        return randomBizCode(len, StringUtils.EMPTY, fieldFun, service);
    }

    /**
     * 用逗号拼接集合
     * @param list 集合
     * @return 字符串
     */
    public static String join (List<String> list){
        return join(list, Function.identity());
    }

    /**
     * 用逗号拼接集合
     * @param list 集合
     * @param mapper 对象
     * @return 字符串
     * @param <T> 集合类型
     */
    public static <T> String join (List<T> list, Function<T, String> mapper){
        StringBuilder msg = new StringBuilder(StringUtils.EMPTY);
        if (CollectionUtils.isNotEmpty(list)) {
            if (!(list instanceof RandomAccess)) {
                list.sort(Comparator.comparing(mapper));
            }
            list.forEach(e -> msg.append(mapper.apply(e)).append(BaseConstants.COMMA));
            delLast(msg);
            return msg.toString();
        }
        return msg.toString();
    }
}
