/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.runner;

import cn.sinozg.applet.common.utils.SpringUtil;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;

/**
 * 设置 可配置的 动态定时任务
 * 配合注解 @RedissonLock 可实现分布式定时调度
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2023
 * @since 2023-10-12 14:37
 */
public abstract class BaseScheduling implements SchedulingConfigurer {
    /** 执行表达式 **/
    protected String cron;

    /** 实现类类型 **/
    @Deprecated
    protected Class<? extends BaseScheduling> clazz;

    /**
     * 设置必须要参数
     */
    public abstract void setParams ();

    /**
     * 执行对应的业务 可以使用注解实现分布式功能<p/>
     */
    public abstract void execute ();

    /**
     * <p>执行</p>
     * @param registrar registrar
     */
    @Override
    public void configureTasks(@NotNull ScheduledTaskRegistrar registrar) {
        initParams();
        Class<? extends BaseScheduling> runClass = clazz != null ? clazz : this.getClass();
        BaseScheduling thisObj = SpringUtil.getBean(runClass);
        registrar.addTriggerTask(thisObj::execute,
                trigger -> {
                    initParams();
                    CronTrigger ct = new CronTrigger(cron);
                    return ct.nextExecution(trigger);
                });
    }

    /**
     * 先设置参数
     */
    private void initParams (){
        setParams();
        if (StringUtils.isBlank(cron)) {
            throw new RuntimeException("请先设置cron参数！");
        }
    }
}
