/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.service;

import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.core.model.RequestParameterEncrypt;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import jakarta.servlet.http.HttpServletRequest;
import java.io.IOException;
import org.apache.commons.lang3.StringUtils;

/**
 * 抽象出密码层 供系统使用，也可以自行实现<br/>
 * 用于 用户密码 系统参数加密解密 签名<br/>
 * 目前两组实现 对称Symmetric加密-非对称Asymmetric加密<br/>
 * ASE&lt;-----&gt;RSA<br/>
 * SM4&lt;-----&gt;SM2<br/>
 * 等
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-12-04 14:06
 */
public interface CipherService {

    /**
     * 加密信息 密码
     * @param plainText 密码
     * @return 加密后的数据
     */
    String encoder(CharSequence plainText);

    /**
     * 密码匹配
     *
     * @param plainText     原始密码
     * @param encodedPassword 加密后的密码
     * @return 是否匹配
     */
    boolean matches(CharSequence plainText, String encodedPassword);

    /**
     * 签名判断 参数排序拼接后 签名再与原始的签名比较
     *
     * @param sha2      是否为 sha2
     * @param signature 签名字符串
     * @param params    要签名的数据
     * @return 是否与签名一致
     */
    boolean signature(boolean sha2, String signature, String... params);

    /**
     * 解密前端传过来的加密数据
     * <p>先用非对称私钥私钥（java）解密前端传过来的对称加密信息，得到前端生成的随机对称加密 key
     * <p>再用 对称加密 key解密得到数据
     * @param content 内容
     * @param asymmetricPrivateKey 非对称私钥
     * @param symmetricEncrypt 对称加密后的数据
     * @return 解密后的数据
     * @throws Exception 异常
     */
    byte[] decryptJson(String content, String asymmetricPrivateKey, String symmetricEncrypt) throws Exception;

    /**
     * 加密数据返回到前端
     *
     * @param request request
     * @param data    请求参数
     * @return 请求参数或者加密后的数据
     */
    default Object encryptJson(HttpServletRequest request, Object data) throws IOException {
        if (PojoWebUtil.nonEncrypt(null, request)) {
            String jsPublicKey = StringUtils.trimToEmpty(request.getHeader(HeaderConstants.X_PUB_KEY));
            String json = JsonUtil.toJson(data);
            if (json != null && StringUtils.isNotBlank(jsPublicKey)) {
                try {
                    return encrypt(json, jsPublicKey);
                } catch (Exception e) {
                    throw new IOException(e);
                }
            }
        }
        return data;
    }

    /**
     * 随机生成 对称加密私钥
     * <p>用对称加密 私钥加密json得到加密信息
     * <p>用前端传过来的 非对称公钥加密 对称加密私钥 得到对称加密私钥加密信息
     *
     * @param content      加密信息
     * @param asymmetricPublicKey 前端传过来的非对称加密公钥
     * @return 加密信息
     */
    RequestParameterEncrypt encrypt(String content, String asymmetricPublicKey) throws Exception;
}
