/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.config;

import cn.sinozg.applet.common.constant.BaseConstants;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.BaseConfig;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.SingleServerConfig;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import jakarta.annotation.Resource;

/**
 * Redisson 配置
 * 有redisson的jar包才加载
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-27 下午 09:52
 **/
@Configuration
@ConditionalOnClass(value = {RedissonClient.class})
public class RedissonConfig {

    @Resource
    private RedisProperties properties;

    @Bean
    public RedissonClient redissonClient() {
        Config config = new Config();
        // 判断是否为集群
        RedisProperties.Cluster cluster = properties.getCluster();
        if (cluster != null && cluster.getNodes() != null) {
            ClusterServersConfig servers = config.useClusterServers();
            for (String nodeAddress : cluster.getNodes()) {
                servers.addNodeAddress(prefixAddress(nodeAddress, null));
            }
            setPassword(servers);
        } else {
            SingleServerConfig single = config.useSingleServer();
            single.setDatabase(properties.getDatabase());
            single.setAddress(prefixAddress(properties.getHost(), properties.getPort()));
            setPassword(single);
        }
        return Redisson.create(config);
    }

    private void setPassword (BaseConfig<?> baseConfig){
        if (StringUtils.isNotBlank(properties.getPassword())) {
            baseConfig.setPassword(properties.getPassword());
        }
    }

    @NotNull
    private String prefixAddress(String host, Integer port) {
        StringBuilder address = new StringBuilder("redis://");
        address.append(host);
        if (port != null) {
            address.append(BaseConstants.COLON).append(port);
        }
        return address.toString();
    }
}