/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.config;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.ThreadFun;
import cn.sinozg.applet.common.handler.ThreadPoolExecutorHandler;
import cn.sinozg.applet.common.utils.ThreadPool;
import com.alibaba.ttl.threadpool.TtlExecutors;
import org.springframework.aop.interceptor.AsyncUncaughtExceptionHandler;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;

import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;


/**
 * 线程池配置 包装注解@Async 支持异步获取信息
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:09
 **/
@EnableAsync
@Configuration
public class ThreadPoolConfig implements AsyncConfigurer {

    /**
     * 配置注解@Async的
     * @return 线程池
     */
    @Override
    public Executor getAsyncExecutor() {
        return TtlExecutors.getTtlExecutorService(new ThreadPoolExecutorHandler("system_async_pool"));
    }

    @Override
    public AsyncUncaughtExceptionHandler getAsyncUncaughtExceptionHandler() {
        return AsyncConfigurer.super.getAsyncUncaughtExceptionHandler();
    }

    /**
     * 执行周期性或定时任务
     * @return 线程池
     */
    @Bean(name = BaseConstants.SCHEDULED_BEAN)
    protected ScheduledExecutorService scheduledExecutorService() {
        return new ScheduledThreadPoolExecutor(ThreadFun.CORE_POOL_SIZE, ThreadFun.threadFactory("schedule-pool")) {
            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                ThreadPool.printException(r, t);
            }
        };
    }
}
