/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;


import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.core.model.IpAddressInfo;
import cn.sinozg.applet.common.core.model.UserAgentInfo;
import com.blueconic.browscap.Capabilities;
import jakarta.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;

/**
 * 操作系统相关判断
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:04
 **/
public class OsUtil {

    private static final Logger log = LoggerFactory.getLogger(OsUtil.class);
    /** IP地址查询 */
    public static final String IP_URL = "https://whois.pconline.com.cn/ipJson.jsp?ip=%s&json=true";

    /** 未知地址 */
    private static final String UNKNOWN = Capabilities.UNKNOWN_BROWSCAP_VALUE;

    /** 本地ip */
    private static final String LOCAL_HOST = "127.0.0.1";

    private OsUtil(){
    }

    /**
     * 判断操作系统
     * @return 是否为windows
     */
    public static boolean isWindows() {
        ServiceEnums service = ServiceEnums.serviceEnums();
        return service == ServiceEnums.WIN;
    }

    /**
     * 通过网络方式获取到ip的地址
     * <p>先从缓存获取，再从网络获取 缓存7天
     * @param ip ip地址
     * @return 实际物理地址
     */
    public static String realAddressByIp (String ip) {
        String ipKey = String.format(BaseRedisKeys.IP_LOCATION, Strings.CS.replace(ip, BaseConstants.SPOT, BaseConstants.UNDERLINE));
        String location = RedisUtil.getCacheObject(ipKey);
        if (StringUtils.isBlank(location)) {
            location = getRealAddressByIp(ip);
            RedisUtil.setCacheObject(ipKey, location, Duration.ofDays(7));
        }
        return location;
    }

    /**
     * 通过网络方式获取到ip的地址
     * @param ip ip地址
     * @return 实际物理地址
     */
    public static String getRealAddressByIp(String ip) {
        // 内网不查询
        if (internalIp(ip)) {
            return "内网IP";
        }
        try {
            byte[] bs = HttpUtil.getBytes(String.format(IP_URL, ip));
            IpAddressInfo address = null;
            if (bs != null) {
                String rspStr = new String(bs, Charset.forName(BaseConstants.GBK));
                address = JsonUtil.toPojo(rspStr, IpAddressInfo.class);
            }
            if (address == null) {
                log.error("远程获取地理位置异常 {}", ip);
                return UNKNOWN;
            }
            String location;
            if (StringUtils.isAllBlank(address.getPro(), address.getCity())) {
                location = address.getAddr();
            } else if (Strings.CS.equals(address.getPro(), address.getCity())) {
                location = address.getPro();
            } else {
                location = String.format("%s %s", address.getPro(), address.getCity());
            }
            return StringUtils.trim(location);
        } catch (Exception e) {
            log.error("获取地理位置异常 {}", ip);
        }
        return UNKNOWN;
    }

    /**
     * 获取id
     * @return ip
     */
    public static String getIpAddr() {
        return getIpAddr(WebUtil.request());
    }

    /**
     * 获取ip地址
     * @param request http request
     * @return ip地址
     */
    public static String getIpAddr(HttpServletRequest request) {
        String unknown = Capabilities.UNKNOWN_BROWSCAP_VALUE;
        if (request == null) {
            return unknown;
        }
        String[] headers = { "x-forwarded-for", "Proxy-Client-IP", "WL-Proxy-Client-IP", "HTTP_Client_IP", "HTTP_X_FORWARDED_FOR" };
        String ip = null;
        int max = headers.length + 1;
        for (int i = 0; i < max; i++) {
            if (StringUtils.isEmpty(ip) || unknown.equalsIgnoreCase(ip)) {
                if (i == headers.length) {
                    ip = request.getRemoteAddr();
                } else {
                    ip = request.getHeader(headers[i]);
                }
            } else {
                break;
            }
        }
        // 使用代理，则获取第一个IP地址
        int maxLen = 15;
        if (StringUtils.isNotEmpty(ip) && ip.length() > maxLen) {
            ip = StringUtils.substringBefore(ip, BaseConstants.COMMA);
        }
        String[] ips = {LOCAL_HOST, "0:0:0:0:0:0:0:1"};
        if (Strings.CS.equalsAny(ip, ips)) {
            ip = getHostIp();
        }
        return ip;
    }

    /**
     * 判断是否为内网ip
     * @param ip ip地址
     * @return  是否为内网ip
     */
    public static boolean internalIp(String ip) {
        byte[] addr = textToNumericFormatV4(ip);
        return internalIp(addr) || LOCAL_HOST.equals(ip);
    }

    /**
     * ip 和mac 地址 集合
     * @return 返回ip mac
     * @throws Exception 异常
     */
    public static ImmutablePair<List<String>, List<String>> ipAndMacAddress () throws Exception {
        List<InetAddress> list = getLocalAllInetAddress();
        List<String> ip = address(list, InetAddress::getHostAddress);
        List<String> mac = address(list, OsUtil::getMacByInetAddress);
        return ImmutablePair.of(ip, mac);
    }

    /**
     * 获取cpu和主板的序列号<br/>
     * macos 没有cpu 序列号，取硬件的uuid
     * @return 序列号
     * @throws Exception 异常
     */
    public static ImmutablePair<String, String> cpuAndMainBoardSerial () throws Exception {
        ServiceEnums enums = ServiceEnums.serviceEnums();
        String cpu = serialNo(enums, true);
        String mainBoard = serialNo(enums, false);
        return ImmutablePair.of(cpu, mainBoard);
    }

    /**
     * 获取ip地址
     * @return ip地址
     */
    public static String getHostIp() {
        try {
            return InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            log.error("获取ip地址失败:{}", e.getMessage());
        }
        return LOCAL_HOST;
    }

    /**
     * 获取客户端名字
     * @return 客户端名称
     */
    public static String getHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            log.error("获取host name失败:{}", e.getMessage());
        }
        return Capabilities.UNKNOWN_BROWSCAP_VALUE;
    }

    /**
     * 获取到浏览器和操作系统信息
     * @param request request
     * @return 浏览器和操作系统信息
     */
    public static UserAgentInfo userAgent (HttpServletRequest request){
        UserAgentInfo agent = new UserAgentInfo();
        String userAgent = request.getHeader(HttpHeaders.USER_AGENT);
        Capabilities capabilities = SystemConfig.PARSER.parse(userAgent);
        agent.setBrowser(formatUserAgent(capabilities.getBrowser(), capabilities.getBrowserMajorVersion()));
        agent.setPlatform(formatUserAgent(capabilities.getPlatform(), capabilities.getPlatformVersion()));
        return agent;
    }

    /**
     * 格式化 浏览器或者操作系统的信息
     * @param name 名称
     * @param version 版本
     * @return 格式化后信息
     */
    private static String formatUserAgent(String name, String version){
        String[] searchStrings = {"0", Capabilities.UNKNOWN_BROWSCAP_VALUE};
        if (Strings.CS.equalsAny(version, searchStrings)) {
            return name;
        }
        return String.format("%s(%s)", name, version);
    }

    /**
     * Converts IPv4 address in its textual presentation form
     * into its numeric binary form.
     * @param src a String representing an IPv4 address in standard format
     * @return ip a byte array representing the IPv4 numeric address
     */
    public static byte[] textToNumericFormatV4(String src) {
        byte[] res = new byte[4];
        long tmpValue = 0;
        int currByte = 0;
        boolean newOctet = true;
        int len = src.length();
        int ipLen = 15;
        if (len == 0 || len > ipLen) {
            return null;
        }
        for (int i = 0; i < len; i++) {
            char c = src.charAt(i);
            if (c == '.') {
                if (newOctet || tmpValue < 0 || tmpValue > 0xff || currByte == 3) {
                    return null;
                }
                res[currByte++] = (byte) (tmpValue & 0xff);
                tmpValue = 0;
                newOctet = true;
            } else {
                int digit = Character.digit(c, 10);
                if (digit < 0) {
                    return null;
                }
                tmpValue *= 10;
                tmpValue += digit;
                newOctet = false;
            }
        }
        int min = 4, max = 8;
        if (newOctet || tmpValue < 0 || tmpValue >= (1L << ((min - currByte) * max))) {
            return null;
        }
        res[currByte] = (byte) ((tmpValue >> (24 - 8 * currByte)) & 0xff);
        return res;
    }

    /**
     * 获取某个网络接口的Mac地址
     * @param inetAddr 网络接口
     * @return Mac地址
     */
    public static String getMacByInetAddress(InetAddress inetAddr) {
        try {
            byte[] mac = NetworkInterface.getByInetAddress(inetAddr).getHardwareAddress();
            StringBuilder info = new StringBuilder();
            for (byte b : mac) {
                // 将十六进制byte转化为字符串
                String temp = Integer.toHexString(b & 0xff);
                if (temp.length() == 1) {
                    info.append("0");
                }
                info.append(temp);
                info.append(BaseConstants.MIDDLE_LINE);
            }
            StringUtil.delLast(info);
            return info.toString().toUpperCase();
        } catch (SocketException e) {
            log.error("获取网络接口的Mac地址错误 {}", inetAddr);
        }
        return null;
    }

    /**
     * 判断是否为内网ip
     * <p>A类  10.0.0.0-10.255.255.255
     * <p>B类  172.16.0.0-172.31.255.255
     * <p>C类  192.168.0.0-192.168.255.255
     * @param addr 地址
     * @return 是否为内网ip
     */
    private static boolean internalIp(byte[] addr) {
        int internalLength = 2;
        if (ArrayUtils.isEmpty(addr) || addr.length < internalLength) {
            return true;
        }
        final byte b0 = addr[0];
        final byte b1 = addr[1];
        // 10.x.x.x/8
        final byte section01 = 0x0A;
        // 172.16.x.x/12
        final byte section02 = (byte) 0xAC;
        final byte section03 = (byte) 0x10;
        final byte section04 = (byte) 0x1F;
        // 192.168.x.x/16
        final byte section05 = (byte) 0xC0;
        final byte section06 = (byte) 0xA8;
        switch (b0) {
            case section01:
                return true;
            case section02:
                if (b1 >= section03 && b1 <= section04) {
                    return true;
                }
            case section05:
                if (b1 == section06) {
                    return true;
                }
            default:
                return false;
        }
    }

    /**
     * 根据指令  获取序列号
     * @param enums 枚举值
     * @param cpu 是否为cpu
     * @return 序列号
     * @throws Exception 异常
     */
    private static String serialNo(ServiceEnums enums, boolean cpu) throws Exception {
        String command = cpu ? enums.getCpuCommand() : enums.getMbCommand();
        String substr = cpu ? enums.getCpuKey() : enums.getMbKey();
        Integer index = enums.getIndex();
        Process process = Runtime.getRuntime().exec(command);
        String info;
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()))){
            Stream<String> stream = Stream.empty();
            if (index != null) {
                stream = reader.lines().skip(index);
            } else if (StringUtils.isNotBlank(substr)) {
                stream = reader.lines().filter(line -> line.contains(substr))
                        .map(line -> StringUtils.substringBefore(line, BaseConstants.COLON));
            }
            info = stream.findFirst().orElse(StringUtils.EMPTY).trim();
        }
        if (cpu && enums.getCpuHandle() != null) {
            info = enums.getCpuHandle().apply(info);
        }
        return info;
    }

    /**
     * 获取地址
     * @param list 网络接口地址
     * @param map map
     * @return 地址
     */
    private static List<String> address (List<InetAddress> list, Function<InetAddress, String> map) {
        List<String> result = null;
        if(CollectionUtils.isNotEmpty(list)){
            // 2. 获取所有网络接口的Mac地址
            result = list.stream().map(map).distinct().map(String::toUpperCase).collect(Collectors.toList());
        }
        return result;
    }


    /**
     * 获取当前服务器所有符合条件的InetAddress
     * @return InetAddress
     * @throws Exception 异常
     */
    private static List<InetAddress> getLocalAllInetAddress() throws Exception {
        List<InetAddress> result = new ArrayList<>();
        // 遍历所有的网络接口
        for (Enumeration<NetworkInterface> nie = NetworkInterface.getNetworkInterfaces(); nie.hasMoreElements();) {
            NetworkInterface ni =  nie.nextElement();
            // 在所有的接口下再遍历IP
            for (Enumeration<InetAddress> iae = ni.getInetAddresses(); iae.hasMoreElements(); ) {
                InetAddress ia = iae.nextElement();
                // 排除 LoopbackAddress、SiteLocalAddress、LinkLocalAddress、MulticastAddress类型的IP地址
                if (!ia.isLoopbackAddress() && !ia.isLinkLocalAddress() && !ia.isMulticastAddress()) {
                    result.add(ia);
                }
            }
        }
        return result;
    }
}
