/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.config;

import cn.sinozg.applet.common.utils.DateUtil;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.jsontype.impl.LaissezFaireSubTypeValidator;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalTimeSerializer;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;

/**
 * jackson 基本配置
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 10:08
 **/
@Configuration
public class JacksonConfig {

    @Bean
    public JsonMapper jsonMapper() {
        JsonMapper jsonMapper = JsonMapper.builder().build();
        initConfig(jsonMapper, false, true, true);
        return jsonMapper;
    }

    /**
     * redis 配置的格式化
     * @param jsonMapper 对象
     */
    public static void mapperConfig(JsonMapper jsonMapper){
        initConfig(jsonMapper, true, true, true);
    }
    /**
     * 初始化配置
     * @param jsonMapper 对象
     * @param config 基础信息
     * @param simple 数字类型
     * @param date 日期类型
     */
    public static void initConfig(JsonMapper jsonMapper, boolean config, boolean simple, boolean date){
        initConfig(jsonMapper, config, simple, date, false);
    }

    /**
     * 初始化配置
     * @param jsonMapper 对象
     * @param config 基础信息
     * @param simple 数字类型
     * @param date 日期类型
     * @param disableUnknown 忽略json中不识别的属性
     */
    public static void initConfig(JsonMapper jsonMapper, boolean config, boolean simple, boolean date, boolean disableUnknown){
        if (config) {
            initConfigModule(jsonMapper);
        }
        if (simple) {
            initSimpleModule(jsonMapper);
        }
        if (date) {
            initDateModule(jsonMapper);
        }
        // 忽略json中不识别的属性
        jsonMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, disableUnknown);
    }

    /**
     * 基本配置
     * @param jm 对象
     */
    private static void initConfigModule(JsonMapper jm){
        // 忽略无法转换的对象
        jm.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        // 过滤掉为null的字段
        jm.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        jm.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        jm.activateDefaultTyping(LaissezFaireSubTypeValidator.instance, JsonMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.WRAPPER_ARRAY);
    }

    /**
     * 等基础数据 初始化
     * @param jsonMapper jsonMapper
     */
    private static void initSimpleModule (JsonMapper jsonMapper){
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(Long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(Long.TYPE, ToStringSerializer.instance);
        simpleModule.addSerializer(long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(BigDecimal.class, ToStringSerializer.instance);
        jsonMapper.registerModule(simpleModule);
        jsonMapper.enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS);
    }
    /**
     * 日期 初始化
     * @param jsonMapper jsonMapper
     */
    private static void initDateModule(JsonMapper jsonMapper){
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(DateUtil.formatter(DateUtil.YYYY_MM_DD_HH_MM_SS)));
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(DateUtil.formatter(DateUtil.YYYY_MM_DD)));
        javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(DateUtil.formatter(DateUtil.HH_MM_SS)));
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(DateUtil.formatter(DateUtil.YYYY_MM_DD_HH_MM_SS)));
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer(DateUtil.formatter(DateUtil.YYYY_MM_DD)));
        javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer(DateUtil.formatter(DateUtil.HH_MM_SS)));
        jsonMapper.registerModule(javaTimeModule).registerModule(new ParameterNamesModule());
    }
}
