/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.service.impl;

import cn.sinozg.applet.common.constant.CipherConstants;
import cn.sinozg.applet.common.core.model.RequestParameterEncrypt;
import cn.sinozg.applet.common.service.CipherService;
import cn.sinozg.applet.common.utils.SmUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;

import java.util.Arrays;

/**
 * 国密实现 系统加密
 * 使用sm3 签名
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-12-04 14:27
 */
@Service
@ConditionalOnProperty(prefix = CipherConstants.PREFIX, name = CipherConstants.TYPE_NAME, havingValue = CipherConstants.TYPE_SM)
public class CipherSmServiceImpl implements CipherService {

    private static final Logger log = LoggerFactory.getLogger(CipherSmServiceImpl.class);

    @Override
    public String encoder(CharSequence plainText) {
        return SmUtil.encryptSm3(plainText.toString());
    }

    @Override
    public boolean matches(CharSequence plainText, String encodedPassword) {
        return SmUtil.matchesSm3(plainText.toString(), encodedPassword);
    }

    @Override
    public boolean signature(boolean sha2, String signature, String... params) {
        if (StringUtils.isBlank(signature)) {
            log.error("非法请求参数，签名参数为空 : {}", signature);
            return false;
        }
        if (StringUtils.isAnyEmpty(params)) {
            log.error("非法请求参数，有部分参数为空 : {}", Arrays.toString(params));
            return false;
        }
        Arrays.sort(params);
        String source = StringUtils.join(params);
        return signature.equals(SmUtil.encryptSm3(source));
    }

    /**
     * <p>先用非对称私钥（java）解密前端传过来的 对称加密加密信息，得到前端生成的随机对称加密 key
     * <p>再用 对称加密  key解密得到数据
     * @param content 内容
     * @param asymmetricPrivateKey 非对称私钥
     * @param symmetricEncrypt 对称加密后的数据
     * @return 解密后的数据
     * @throws Exception 异常
     */
    @Override
    public byte[] decryptJson(String content, String asymmetricPrivateKey, String symmetricEncrypt) throws Exception {
        String symmetricKey = SmUtil.decryptSm2(symmetricEncrypt, asymmetricPrivateKey);
        return SmUtil.decryptSm4(content, symmetricKey);
    }

    /**
     * 随机生成16位的 对称加密私钥
     * <p>用对称加密私钥 加密json得到加密信息
     * <p>用前端传过来的 RSA 公钥加密 aes私钥 得到aes私钥加密信息
     *
     * @param content      加密信息
     * @param asymmetricPublicKey 前端传过来的非对称加密公钥
     * @return 加密信息
     */

    @Override
    public RequestParameterEncrypt encrypt(String content, String asymmetricPublicKey) throws Exception {
        // 随机生成 对称私钥
        String symmetricPrivateKey = SmUtil.generateKeySm4();
        // 用对称加密秘钥加密实际的json内容
        String json = SmUtil.encryptSm4(content, symmetricPrivateKey);
        String symmetricEncrypt = SmUtil.encryptSm2(symmetricPrivateKey, asymmetricPublicKey);
        RequestParameterEncrypt encrypt = new RequestParameterEncrypt();
        encrypt.setData(json);
        encrypt.setSymmetricEncrypt(symmetricEncrypt);
        return encrypt;
    }
}
