/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.I18nName;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-12-11 21:26
 */
public class I18nBaseUtil {

    private static final Map<Locale, Function<I18nName, String>> LOCALE_MAP = new LinkedHashMap<>();

    private static List<Locale> SUPPORT_LOCALES;

    /**
     * 支持的语言
     * @return 语言
     */
    public static List<Locale> supportLocales(){
        if(SUPPORT_LOCALES == null){
            SUPPORT_LOCALES = webSupport();
        }
        return SUPPORT_LOCALES;
    }

    protected static Map<Locale, Function<I18nName, String>> supportLocaleMap (){
        if (MapUtils.isEmpty(LOCALE_MAP)) {
            Map<Locale, Function<I18nName, String>> map = baseLocals();
            List<Locale> locales = supportLocales();
            for (Map.Entry<Locale, Function<I18nName, String>> e : map.entrySet()) {
                if (locales.contains(e.getKey())) {
                    LOCALE_MAP.put(e.getKey(), e.getValue());
                }
            }
        }
        return LOCALE_MAP;
    }

    /**
     * web端支持的语言
     * @return web端支持的语言
     */
    private static List<Locale> webSupport(){
        Set<String> fileNames = FileUtil.resourcesFile("classpath*:i18n/**/*.*", "messages");
        List<Locale> list = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(fileNames)) {
            for (String fileName : fileNames) {
                String name = StringUtils.substringBefore(fileName, BaseConstants.SPOT);
                String[] ns = StringUtils.split(name, BaseConstants.UNDERLINE);
                if (ArrayUtils.isEmpty(ns) || ns.length != 3) {
                    continue;
                }
                Locale locale = new Locale(ns[1], ns[2]);
                list.add(locale);
            }
        }
        return list;
    }

    private static Map<Locale, Function<I18nName, String>> baseLocals (){
        Map<Locale, Function<I18nName, String>> localMap = new LinkedHashMap<>();
        localMap.put(Locale.CHINA, I18nName::getName);
        localMap.put(Locale.US, I18nName::getEnName);
        localMap.put(Locale.FRANCE, I18nName::getFrName);
        // 西班牙语
        localMap.put(new Locale("es", "ES"), I18nName::getEsName);
        // 俄语
        localMap.put(new Locale("ru", "RU"), I18nName::getRuName);
        localMap.put(Locale.JAPAN, I18nName::getJaName);
        return localMap;
    }
}
