/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.service.impl;

import cn.sinozg.applet.common.handler.TenantHandler;
import cn.sinozg.applet.common.interceptor.TenantLicenseInterceptor;
import cn.sinozg.applet.common.properties.ApiUrlValue;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.properties.TenantValue;
import cn.sinozg.applet.common.service.FrameworkTenantService;
import com.baomidou.mybatisplus.extension.plugins.MybatisPlusInterceptor;
import com.baomidou.mybatisplus.extension.plugins.inner.TenantLineInnerInterceptor;
import jakarta.annotation.Resource;
import java.util.List;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;

/**
 * @author xyb
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-11-03 14:42
 */
@Service
public class FrameworkTenantServiceImpl implements FrameworkTenantService {

    @Resource
    private AppValue app;

    @Resource
    private TenantLicenseInterceptor tenantLicenseInterceptor;

    private static final Logger log = LoggerFactory.getLogger(FrameworkTenantServiceImpl.class);

    @Override
    public void dbInterceptor(MybatisPlusInterceptor interceptor) {
        TenantValue tenant = app.getTenant();
        if (tenant.isEnable()) {
            log.info("开启了多租户模式，加载多租户Interceptor！");
            // 多租户
            TenantLineInnerInterceptor tenantInner = new TenantLineInnerInterceptor(new TenantHandler(tenant));
            interceptor.addInnerInterceptor(tenantInner);
        }
    }

    @Override
    public void expiredCheckInterceptor(InterceptorRegistry registry) {
        TenantValue tenant = app.getTenant();
        ApiUrlValue apiUrl = app.getApiUrl();
        if (tenant.isEnable() && tenant.isExpiredCheck()) {
            if (CollectionUtils.isEmpty(apiUrl.getParamsPatterns())) {
                log.error("请先设置请求拦截: app.apiUrl.paramsPatterns");
                return;
            }
            List<String> excludePath = apiUrl.getAuthExcludePath();
            if (CollectionUtils.isNotEmpty(tenant.getCheckExcludePath())) {
                excludePath.addAll(tenant.getCheckExcludePath());
            }
            log.info("开启了多租户授权检测模式，加载多租户InterceptorRegistry！");
            registry.addInterceptor(tenantLicenseInterceptor)
                    .addPathPatterns(apiUrl.getAuthPatterns())
                    .excludePathPatterns(excludePath);
        }
    }
}
