package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.properties.WechatValue;
import com.fasterxml.jackson.databind.JsonNode;
import com.wechat.pay.contrib.apache.httpclient.WechatPayHttpClientBuilder;
import com.wechat.pay.contrib.apache.httpclient.auth.PrivateKeySigner;
import com.wechat.pay.contrib.apache.httpclient.auth.Verifier;
import com.wechat.pay.contrib.apache.httpclient.auth.WechatPay2Credentials;
import com.wechat.pay.contrib.apache.httpclient.auth.WechatPay2Validator;
import com.wechat.pay.contrib.apache.httpclient.cert.CertificatesManager;
import com.wechat.pay.contrib.apache.httpclient.exception.ParseException;
import com.wechat.pay.contrib.apache.httpclient.exception.ValidationException;
import com.wechat.pay.contrib.apache.httpclient.notification.Notification;
import com.wechat.pay.contrib.apache.httpclient.notification.NotificationHandler;
import com.wechat.pay.contrib.apache.httpclient.notification.NotificationRequest;
import com.wechat.pay.contrib.apache.httpclient.util.AesUtil;
import com.wechat.pay.contrib.apache.httpclient.util.PemUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.MediaType;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.PrivateKey;

/**
 * @Author: xyb
 * @Description:
 * @Date: 2023-03-24 下午 08:43
 **/
public class PayUtil {
    private PayUtil(){
    }
    private static volatile PrivateKey PRIVATE_KEY;
    private static final Logger log = LoggerFactory.getLogger(PayUtil.class);
    public static PrivateKey privateKeyInstance() {
        if (PRIVATE_KEY == null) {
            synchronized (PayUtil.class) {
                if (PRIVATE_KEY == null) {
                    Resource resourcePrivateKey = new FileSystemResource(new File(SystemConfig.APP.getWechat().getPrivateKeyPath()));
                    try (InputStream is = resourcePrivateKey.getInputStream()) {
                        PRIVATE_KEY = PemUtil.loadPrivateKey(is);
                    } catch (Exception e) {
                        log.error("加载私钥证书失败！", e);
                    }
                }
            }
        }
        return PRIVATE_KEY;
    }

    /**
     * 初始化  clientBuilder和证书管理器
     * @return CloseableHttpClient
     */
    public static CloseableHttpClient httpClient() {
        WechatValue wechat = SystemConfig.APP.getWechat();
        CloseableHttpClient client = null;
        try {
            // 私钥
            PrivateKey privateKey = privateKeyInstance();
            String mchId = wechat.getMchId();
            String mchSerialNo = wechat.getMchSerialNo();
            Verifier verifier = verifier(wechat);
            // 初始化
            client = WechatPayHttpClientBuilder.create()
                    .withMerchant(mchId, mchSerialNo, privateKey)
                    .withValidator(new WechatPay2Validator(verifier))
                    .build();
        } catch (Exception e) {
            log.error("加载小程序支付证书失败！", e);
            e.printStackTrace();
        }
        return client;
    }
    /**
     * 验证签名
     * @param request 序列号
     * @param body 请求体
     * @return 签名是否正确
     */
    public static boolean signVerify(HttpServletRequest request, String body) throws ValidationException, ParseException {
        String timestamp = request.getHeader(HeaderConstants.TIMESTAMP);
        String nonce = request.getHeader(HeaderConstants.NONCE);
        String signature = request.getHeader(HeaderConstants.SIGNATURE);
        String serial = request.getHeader(HeaderConstants.SERIAL);
        log.info("支付回调参数，{}, {}, {}, {}", timestamp, nonce, signature, serial);
        WechatValue wechat = SystemConfig.APP.getWechat();
        // 构建request，传入必要参数
        NotificationRequest notificationRequest = new NotificationRequest.Builder().withSerialNumber(serial)
                .withNonce(nonce)
                .withTimestamp(timestamp)
                .withSignature(signature)
                .withBody(body)
                .build();
        Verifier verifier = verifier(wechat);
        NotificationHandler handler = new NotificationHandler(verifier, wechat.getApiV3Key().getBytes(StandardCharsets.UTF_8));
        // 验签和解析请求体
        Notification notification = handler.parse(notificationRequest);
        return notification != null;
    }

    /**
     * 获取证书验证器
     * @param wechat 请求参数
     * @return 验证器
     */
    private static Verifier verifier(WechatValue wechat){
        CertificatesManager certificates = CertificatesManager.getInstance();
        // 私钥 加载商户私钥（privateKey：私钥字符串）
        PrivateKey privateKey = privateKeyInstance();
        // 证书 证书管理器
        // 使用自动更新的签名验证器，不需要传入证书
        // mchId：商户号,mchSerialNo：商户证书序列号,apiV3Key：V3密钥
        String mchId = wechat.getMchId();
        try {
            certificates.putMerchant(mchId, new WechatPay2Credentials(mchId, new PrivateKeySigner(wechat.getMchSerialNo(), privateKey)),
                    wechat.getApiV3Key().getBytes(StandardCharsets.UTF_8));
            return certificates.getVerifier(mchId);
        } catch (Exception e) {
            log.error("获取证书验证器失败！", e);
            throw new RuntimeException("获取证书验证器失败！");
        }
    }

    /**
     * 微信post 请求
     * @param url 请求地址
     * @param params 参数
     * @param clazz 返回类型
     * @return 请求结果
     * @param <T> 返回类型
     */
    public static <T> T post (String url, Object params, Class<T> clazz){
        HttpPost post = new HttpPost(url);
        post.setHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_JSON_VALUE);
        post.setHeader(HttpHeaders.CONTENT_TYPE, BaseConstants.APPLICATION_JSON);
        String json = JsonUtil.toJson(params);
        if (json == null) {
            return null;
        }
        String result = null;
        post.setEntity(new StringEntity(json, StandardCharsets.UTF_8));
        try (CloseableHttpClient client = httpClient();
             CloseableHttpResponse response = client.execute(post)) {
            int code = response.getStatusLine().getStatusCode();
            if (code == HttpStatus.SC_OK) {
                result = EntityUtils.toString(response.getEntity());
                // 204 成功但是没有body
            } else if (code == HttpStatus.SC_NO_CONTENT) {
                result = BaseConstants.EMPTY_JSON;
            } else {
                throw new CavException("BIZ000100010");
            }
        } catch (Exception e) {
            log.error("微信支付请求失败！", e);
        }
        if (clazz == String.class) {
            return PojoUtil.cast(result);
        }
        return JsonUtil.toPojo(result, clazz);
    }


    /**
     * 解密订单信息
     * @param body 应答报文主体
     * @return 解密结果
     */
    public static String decryptOrder(String body) {
        WechatValue wechat = SystemConfig.APP.getWechat();
        try {
            AesUtil aesUtil = new AesUtil(wechat.getApiV3Key().getBytes(StandardCharsets.UTF_8));
            JsonNode node = JsonUtil.toNode(body);
            JsonNode resource = node.get("resource");
            String ciphertext = resource.get("ciphertext").textValue();
            String associatedData = resource.get("associated_data").textValue();
            String nonce = resource.get("nonce").textValue();
            return aesUtil.decryptToString(associatedData.getBytes(StandardCharsets.UTF_8), nonce.getBytes(StandardCharsets.UTF_8), ciphertext);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return StringUtils.EMPTY;
    }
}