/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.biz.system.service.impl;

import cn.sinozg.applet.biz.system.service.FrameworkPayService;
import cn.sinozg.applet.biz.system.service.PaymentService;
import cn.sinozg.applet.biz.system.vo.request.OrderPaymentAmount;
import cn.sinozg.applet.biz.system.vo.request.OrderPaymentCloseRequest;
import cn.sinozg.applet.biz.system.vo.request.OrderPaymentPayer;
import cn.sinozg.applet.biz.system.vo.request.OrderPaymentRequest;
import cn.sinozg.applet.biz.system.vo.request.OrderRefundAmount;
import cn.sinozg.applet.biz.system.vo.request.OrderRefundRequest;
import cn.sinozg.applet.biz.system.vo.request.WebPaymentRequest;
import cn.sinozg.applet.biz.system.vo.request.WebRefundRequest;
import cn.sinozg.applet.biz.system.vo.response.OrderPaymentResponse;
import cn.sinozg.applet.biz.system.vo.response.OrderRefundResponse;
import cn.sinozg.applet.biz.system.vo.response.PaymentCallbackDetail;
import cn.sinozg.applet.biz.system.vo.response.PaymentCallbackResponse;
import cn.sinozg.applet.biz.system.vo.response.RefundCallbackDetail;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.LoginUserVo;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.properties.WechatValue;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.PayUtil;
import cn.sinozg.applet.common.utils.UserUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import java.io.BufferedReader;

/**
 * @Author: xyb
 * @Description:
 * @Date: 2023-03-22 下午 06:40
 **/
@Service
public class PaymentServiceImpl implements PaymentService {

    @Resource
    private AppValue appValue;

    @Resource
    private FrameworkPayService payService;
    private final Logger log = LoggerFactory.getLogger(PaymentServiceImpl.class);
    @Override
    public String payment (WebPaymentRequest param) {
        LoginUserVo user = UserUtil.user();
        WechatValue wechat = appValue.getWechat();
        OrderPaymentRequest payRequest = generatePayRequest(param, user.getOpenId());

        OrderPaymentResponse result = PayUtil.post(wechat.getUnifiedOrderUrl(), payRequest, OrderPaymentResponse.class);

        if (result == null || StringUtils.isEmpty(result.getPrepayId())) {
            throw new CavException("BIZ000100007");
        }
        return result.getPrepayId();
    }

    @Override
    public boolean closeOrder(String outTradeNo) {
        WechatValue wechat = appValue.getWechat();
        OrderPaymentCloseRequest request = new OrderPaymentCloseRequest();
        request.setMchId(wechat.getMchId());
        String url = String.format(wechat.getCloseUrl(), outTradeNo);
        String result = PayUtil.post(url, request, String.class);
        return StringUtils.equals(BaseConstants.EMPTY_JSON, result);
    }

    @Override
    public PaymentCallbackResponse paymentCallback(HttpServletRequest request){
        log.info("支付回调..................");
        PaymentCallbackResponse response = new PaymentCallbackResponse();
        response.setCode(BaseConstants.FA);
        String body = body(request);
        try {
            log.info("支付回调报文信息：{}", body);
            boolean flag = PayUtil.signVerify(request, body);
            if (!flag) {
                throw new RuntimeException("支付校验失败！");
            }
            String json = PayUtil.decryptOrder(body);
            log.info("回调后的解密消息，{}", json);
            PaymentCallbackDetail detail = JsonUtil.toPojo(json, PaymentCallbackDetail.class);
            if (detail == null) {
                throw new RuntimeException("解析支付参数失败！");
            }
            payService.paymentCallback(detail.getOutTradeNo(), BaseConstants.SUC.equals(detail.getTradeState()), detail);
            response.setCode(BaseConstants.SUC);
            response.setMessage("成功！");
        } catch (Exception e) {
            log.error("支付回调 处理回调发生异常！", e);
            response.setMessage(e.getMessage());
        }
        return response;
    }

    @Override
    public OrderRefundResponse refund (WebRefundRequest params){
        WechatValue wechat = appValue.getWechat();
        OrderRefundRequest request = new OrderRefundRequest();
        request.setTransactionId(params.getTransactionId());
        request.setOutRefundNo(params.getRefundNo());
        request.setFundsAccount("AVAILABLE");
        request.setNotifyUrl(wechat.getRefundCallbackUrl());
        request.setReason(params.getRefundDesc());

        OrderRefundAmount amount = new OrderRefundAmount();
        amount.setTotal(params.getTotalFee());
        amount.setRefund(params.getRefundFee());
        request.setAmount(amount);

        OrderRefundResponse refundInfo = PayUtil.post(wechat.getRefundUrl(), request, OrderRefundResponse.class);
        if (refundInfo == null || !BaseConstants.SUC.equals(refundInfo.getStatus())) {
            throw new CavException("BIZ000100008");
        }
        // 回写订单信息
        return refundInfo;
    }


    @Override
    public PaymentCallbackResponse refundCallBack(HttpServletRequest request) {
        log.info("退款回调..................");
        PaymentCallbackResponse response = new PaymentCallbackResponse();
        response.setCode(BaseConstants.FA);
        String body = body(request);
        try {
            log.info("退款回调报文信息：{}", body);
            boolean flag = PayUtil.signVerify(request, body);
            if (!flag) {
                throw new RuntimeException("退款校验失败！");
            }
            String json = PayUtil.decryptOrder(body);
            log.info("退款回调后的解密消息，{}", json);
            RefundCallbackDetail detail = JsonUtil.toPojo(json, RefundCallbackDetail.class);
            if (detail == null) {
                throw new RuntimeException("解析退款参数失败！");
            }
            boolean b = payService.refundCallback(detail);
            if (!b) {
                throw new RuntimeException("更新业务数据失败！");
            }
            response.setCode(BaseConstants.SUC);
            response.setMessage("成功！");
        } catch (Exception e) {
            log.error("退款回调 处理回调发生异常！", e);
            response.setMessage(e.getMessage());
        }
        return response;
    }

    /**
     * 获取到请求体
     * @param request HttpServletRequest
     * @return 请求参数
     */
    private String body (HttpServletRequest request) {
        String temp;
        StringBuilder body = new StringBuilder();
        try (BufferedReader br = request.getReader()) {
            while ((temp = br.readLine()) != null) {
                body.append(temp);
            }
        } catch (Exception e) {
            log.error("获取请求体失败！", e);
        }
        return body.toString();
    }

    /**
     * 组装微信支付请求参数
     * @param params 支付请求
     * @return 微信端需要的参数
     */
    private OrderPaymentRequest generatePayRequest(WebPaymentRequest params, String openId) {
        WechatValue wechat = appValue.getWechat();
        OrderPaymentRequest request = new OrderPaymentRequest();
        request.setAppId(wechat.getAppid());
        request.setMchId(wechat.getMchId());
        request.setNotifyUrl(wechat.getNotifyUrl());
        request.setDescription(params.getDescription());
        request.setOutTradeNo(params.getOrderId());

        OrderPaymentAmount amount = new OrderPaymentAmount();
        amount.setTotal(params.getTotalFee());
        request.setAmount(amount);

        OrderPaymentPayer payer = new OrderPaymentPayer();
        payer.setOpenId(openId);
        request.setPayer(payer);
        return request;
    }

}