/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.tool;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.tool.model.JavaFileInfo;
import cn.sinozg.applet.tool.model.RepeatClassRequest;
import cn.sinozg.applet.tool.model.RepeatClassResponse;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 统计多余的文件
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-07-08 22:52
 */
public class RepeatClassFile {
    public static final String SUFFIX = ".java";

    private static final AtomicInteger AI = new AtomicInteger(0);
    private static final Pattern PATTERN = Pattern.compile("message = \"\\{([^}]*)}\"");
    private static final Logger log = LoggerFactory.getLogger(RepeatClassFile.class);

    /**
     * 读取项目中多余的文件
     * @param params 参数
     * @return 返回结果
     * @throws IOException 异常
     */
    public static RepeatClassResponse absentClass (RepeatClassRequest params) throws IOException {
        File file = new File(params.getRootPath());
        RepeatClassResponse response = new RepeatClassResponse();
        Map<String, JavaFileInfo> map = new TreeMap<>();
        itemFile(map, file);
        int itemNum = 0;
        String[] defaultExistPackage = PojoUtil.toArray(params.getDefaultExistPackage(), String.class);
        String[] defaultExcludePackage = PojoUtil.toArray(params.getDefaultExcludePackage(), String.class);
        out : for (Map.Entry<String, JavaFileInfo> e : map.entrySet()) {
            // biz common 包下的文件检查 公共包
            if (Strings.CS.containsAny(e.getKey(), defaultExistPackage) &&
                    !Strings.CS.containsAny(e.getKey(), defaultExcludePackage)
            ) {
                itemNum ++;
                List<String> lines = e.getValue().getLines();
                if (CollectionUtils.isNotEmpty(lines)) {
                    for (String line : lines) {
                        if (Strings.CS.startsWithAny(line, "@Component")) {
                            continue out;
                        }
                    }
                }
                boolean exit = false;
                String ln = StringUtils.substringAfterLast(e.getKey(), BaseConstants.SPOT);
                String packageName = StringUtils.substringBeforeLast(e.getKey(), BaseConstants.SPOT);
                String importClazz = "import " + e.getKey() + ";";
                for (Map.Entry<String, JavaFileInfo> om : map.entrySet()) {
                    // 自己跳过
                    if (om.getKey().equals(e.getKey())) {
                        continue;
                    }
                    // 在同一个包下面的话 只判断类 否则判断包import
                    if (om.getValue().getContext().contains(importClazz)) {
                        exit = true;
                        break;
                    } else {
                        String pm = StringUtils.substringBeforeLast(om.getKey(), BaseConstants.SPOT);
                        if (packageName.equals(pm) && Strings.CS.containsAny(om.getValue().getContext(), StringUtils.SPACE + ln + StringUtils.SPACE, "<" + ln + ">")) {
                            exit = true;
                            break;
                        }
                    }
                }
                if (!exit) {
                    response.addNonQuote(e.getKey());
                }
                schema(e.getKey(), response);
            }
        }
        response.setTotalFileSize(AI.get());
        response.setHandleFileSize(map.size());
        response.setJavaFileSize(itemNum);
        if (StringUtils.isNotBlank(params.getValidatorFile())) {
            validator(map, params.getValidatorFile(), response);
        }
        return response;
    }

    private static void validator (Map<String, JavaFileInfo> map, String vpath, RepeatClassResponse response){
        Map<String, String> vmap = new HashMap<>();
        try {
            List<String> lines = FileUtils.readLines(new File(vpath), Charset.defaultCharset());
            for (String line : lines) {
                if (Strings.CS.contains(line, "=")) {
                    String[] kv = StringUtils.split(line, "=");
                    vmap.put(kv[0], kv[1]);
                }
            }
            JavaFileInfo info;
            String v;
            for (Map.Entry<String, JavaFileInfo> e : map.entrySet()) {
                info = e.getValue();
                lines = info.getLines();
                for (String line : lines) {
                    Matcher matcher = PATTERN.matcher(line);
                    if (matcher.find()) {
                        v = matcher.group(1);
                        vmap.remove(v);
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        for (Map.Entry<String, String> e : vmap.entrySet()) {
            response.addSurplusValidatorCode(e.getKey());
        }
    }

    private static void itemFile (Map<String, JavaFileInfo> map, File file) throws IOException {
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files != null) {
                for (File f : files) {
                    itemFile(map, f);
                }
            }
        } else {
            String name = file.getAbsolutePath();
            if (Strings.CS.containsAny(name, "target", ".git")) {
                return;
            }
            AI.incrementAndGet();
            if (Strings.CS.endsWith(name, SUFFIX)) {
                name = StringUtils.substringAfter(name,  File.separator + "java" + File.separator);
                name = Strings.CS.replace(name, SUFFIX, StringUtils.EMPTY);
                name = Strings.CS.replace(name, File.separator, BaseConstants.SPOT);
                JavaFileInfo fileInfo = new JavaFileInfo();
                List<String> lines = FileUtils.readLines(file, Charset.defaultCharset());
                fileInfo.setLines(lines);
                fileInfo.setContext(String.join(StringUtils.EMPTY, lines));
                map.put(name, fileInfo);
            }
        }
    }


    private static void schema (String name, RepeatClassResponse response){
        if (Strings.CS.containsAny(name, ".handle.", ".service.", ".utils.")) {
            return;
        }
        try {
            Class<?> clazz = Class.forName(name);
            Schema schema = clazz.getAnnotation(Schema.class);
            if (schema != null) {
                String schemaName = schema.name();
                if (StringUtils.isNotBlank(schemaName)) {
                    if (!Strings.CS.endsWith(name, BaseConstants.SPOT + schemaName)) {
                        response.addDifferentSchema(name);
                    }
                }
            }
        } catch (Exception e) {
            response.addErrorSchema(name + "<--->" + Arrays.toString(e.getStackTrace()));
        }
    }
}
