/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.tool.i18n;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.I18nName;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.tool.enums.I18nTp;
import cn.sinozg.applet.tool.enums.SheetEnum;
import cn.sinozg.applet.tool.model.DictI18n;
import cn.sinozg.applet.tool.model.I18nInfo;
import cn.sinozg.applet.tool.model.I18nSetGet;
import cn.sinozg.applet.tool.model.ReadConfigParams;
import cn.sinozg.applet.tool.model.WriteConfigParams;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.text.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 国际化 vue 和java 读取本地文件 翻译后 写成其他语言的文件
 * @author xyb
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-11-07 23:17
 */
public class I18nReadWrite {

    private static final String EQ = "=";

    private static final String MSG = "messages";

    private static final String VALIDATOR = "validator";

    private static final String TITLE = "title";

    private static final Map<String, String> MAP = new TreeMap<>();

    private static final Pattern PATTERN = Pattern.compile("\"[A-Za-z0-9]+\" :");

    private static final Logger log = LoggerFactory.getLogger(I18nReadWrite.class);

    public static final Map<I18nTp, I18nSetGet> MAPPER = new LinkedHashMap<>();

    static {
        MAPPER.put(I18nTp.CN, new I18nSetGet(I18nInfo::getCn, I18nInfo::setCn, I18nName::setName));
        MAPPER.put(I18nTp.EN, new I18nSetGet(I18nInfo::getEn, I18nInfo::setEn, I18nName::setEnName));
        MAPPER.put(I18nTp.ES, new I18nSetGet(I18nInfo::getEs, I18nInfo::setEs, I18nName::setEsName));
        MAPPER.put(I18nTp.FR, new I18nSetGet(I18nInfo::getFr, I18nInfo::setFr, I18nName::setFrName));
        MAPPER.put(I18nTp.JA, new I18nSetGet(I18nInfo::getJa, I18nInfo::setJa, I18nName::setJaName));
    }

    /**
     * 读取配置文件转为excel数据
     * @param params 参数
     * @return 数据
     * @param <T> 类型
     */
    public static <T> List<List<T>> readConfig (ReadConfigParams<T> params){
        List<List<T>> data = new ArrayList<>();
        Class<T> clazz = params.getOut();
        // java
        Map<String, List<String>> config = readJavaI18n();
        Map<String, List<I18nInfo>> map = readJavaI18n(config, true);
        for (Map.Entry<String, List<I18nInfo>> e : map.entrySet()) {
            data.add(PojoUtil.copyList(e.getValue(), clazz));
        }
        // vue
        if (StringUtils.isNotBlank(params.getVuePath())) {
            List<I18nInfo> vue = readVue(params.getVuePath());
            data.add(PojoUtil.copyList(vue, clazz));
        }
        // dict
        if (params.getDict() != null) {
            List<DictI18n> dictList = params.getDict().get();
            List<I18nInfo> dict = readDict(dictList);
            data.add(PojoUtil.copyList(dict, clazz));
        }
        // app
        if (StringUtils.isNotBlank(params.getAppPath())) {
            map = readApp(params.getAppPath());
            for (Map.Entry<String, List<I18nInfo>> e : map.entrySet()) {
                data.add(PojoUtil.copyList(e.getValue(), clazz));
            }
        }
        return data;
    }

    /**
     * 打印excel 表格的国际化数据
     * @param params 参数
     */
    public static void importExcel(WriteConfigParams params) {
        SheetEnum sheet = params.getSheet();
        List<I18nInfo> list = params.getData();
        if (sheet == SheetEnum.DICT) {
            updateDict(params);
        } else if (sheet == SheetEnum.VUE) {
            Map<String, String> map = writeToVue(params.getData(), params.getI18nTp(), params.getVuePath());
            map.forEach((k, v) -> {
                System.out.println(k);
                System.out.println("-----------------------------------");
                System.out.println(v);
                System.out.println();
                System.out.println();
                System.out.println();
                System.out.println();
            });
            // 处理路由
            printRoute(params);
        }  else if (sheet == SheetEnum.APP) {
            Map<String, String> map = new HashMap<>();
            Function<I18nInfo, String> function = getFun(params.getI18nTp());
            for (I18nInfo i : list) {
                map.put(i.getKey(), function.apply(i));
            }
            System.out.println(JsonUtil.toJson(map));
        } else {
            List<String> lines = writeJavaWithCn(list, sheet.getType(), params.getI18nTp());
            String text = String.join("\n", lines);
            System.out.println(text);
        }
    }

    /**
     * 读取标准数据
     * @param list 标准数据
     * @return 可以导出的数据
     */
    private static List<I18nInfo> readDict (List<DictI18n> list){
        List<I18nInfo> infos = new ArrayList<>();
        for (DictI18n dict : list) {
            String key = dict.getType() + BaseConstants.COLON + dict.getValue();
            I18nInfo info = new I18nInfo();
            info.setKey(key);
            I18nName name = dict.getName();
            info.setCn(name.getName());
            info.setEn(name.getEnName());
            info.setJa(name.getJaName());
            info.setEs(name.getEsName());
            infos.add(info);
        }
        return infos;
    }

    /**
     * 更新数据库的标准数据
     * @param params 参数
     */
    private static void updateDict (WriteConfigParams params){
        Consumer<DictI18n> consumer = params.getDictUpdate();
        for (I18nInfo info : params.getData()) {
            DictI18n d = new DictI18n();
            I18nName name = new I18nName();
            name.setName(info.getCn());
            name.setEnName(info.getEn());
            name.setEsName(info.getEs());
            name.setJaName(info.getJa());
            d.setName(name);
            String[] ks = StringUtils.split(info.getKey(), BaseConstants.COLON);
            d.setType(ks[0]);
            d.setValue(ks[1]);
            if (consumer != null) {
                consumer.accept(d);
            }
        }
        Supplier<Boolean> supplier = params.getDictRoot();
        if (supplier != null) {
            supplier.get();
        }
    }

    /**
     * 以中文为模板打印
     * @param list 数据集合
     * @param type 数据类型，验证 消息等
     * @param tp 要写的语言
     * @return 返回的配置
     */
    private static List<String> writeJavaWithCn(List<I18nInfo> list, String type, I18nTp tp){
        Map<String, List<String>> config = readJavaI18n();
        Map<String, String> values = new HashMap<>();
        Function<I18nInfo, String> fun = getFun(tp);
        list.forEach(i -> values.put(i.getKey(), fun.apply(i)));
        List<String> configs = config.get(type + "_zh");
        List<String> result = new ArrayList<>();
        String key;
        for (String line : configs) {
            String nl = line;
            if (Strings.CS.contains(line, EQ)) {
                String[] vs = StringUtils.split(line, EQ);
                if (vs != null && vs.length == 2) {
                    key = vs[0];
                    nl = key + EQ + values.get(key);
                }
            }
            result.add(nl);
        }
        return result;
    }


    /**
     * 写vue 配置文件
     * @param list 结合
     * @param tp 类型
     * @return 模块和对应的 json 公共模块key 为app
     */
    private static Map<String, String> writeToVue (List<I18nInfo> list, I18nTp tp, String path){
        Map<String, Object> map = new HashMap<>();
        Map<String, String> jsJson = new HashMap<>();
        Function<I18nInfo, String> fun = getFun(tp);
        for (I18nInfo ii : list) {
            setMap(ii, fun, map);
        }
        // 将map 变为和原文档的数据结构顺序一致
        Map<String, Object> oldMap = i18nValueMap(path).get(I18nTp.CN);
        recursionSet(oldMap, map);
        Object v;
        Iterator<Map.Entry<String, Object>> it = oldMap.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, Object> e = it.next();
            v = e.getValue();
            if (firstValueIsMap(v)) {
                // 输出
                jsJson.put(e.getKey(), toJsJson(v));
                it.remove();
            }
        }
        jsJson.put(BaseConstants.APP_PREFIX, toJsJson(oldMap));
        return jsJson;
    }

    /**
     * 读取本地vue 国际化配置文件
     * @param path 文件目录
     * @return 国际化信息
     */
    private static List<I18nInfo> readVue(String path){
        Map<I18nTp, Map<String, Object>> valueMap = i18nValueMap(path);
        Map<String, I18nInfo> valuesMap = new TreeMap<>();
        for (Map.Entry<I18nTp, Map<String, Object>> e : valueMap.entrySet()) {
            toMap(e.getValue(), StringUtils.EMPTY, setFun(e.getKey()), valuesMap);
        }
        List<I18nInfo> list = new ArrayList<>();
        valuesMap.forEach((k, v) -> list.add(v));
        return list;
    }

    /**
     * 读取app的国际化配置
     * @param path 路径
     * @return 配置
     */
    private static Map<String, List<I18nInfo>> readApp(String path){
        File file = new File(path);
        File[] fs = file.listFiles();
        Map<String, List<String>> result = new TreeMap<>();
        try {
            if (fs != null) {
                for (File f : fs) {
                    String fm = f.getName();
                    if (Strings.CI.equals(fm, ".json")) {
                        I18nTp tp = I18nTp.contains(fm);
                        String json = FileUtils.readFileToString(f, Charset.defaultCharset());
                        Map<String, Object> map = JsonUtil.toMap(json);
                        List<String> value = new ArrayList<>();
                        for (Map.Entry<String, Object> e : map.entrySet()) {
                            value.add(e.getKey() + EQ + e.getValue());
                        }
                        result.put("app_" + tp.getCode(), value);
                    }
                }
            }
        } catch (Exception e) {
            throw new CavException(e);
        }
        return readJavaI18n(result, false);
    }

    private static void printRoute(WriteConfigParams params) {
        Function<I18nInfo, String> get = getFun(params.getI18nTp());
        String prefix = "route.";
        Map<String, String> mapper = new HashMap<>();
        params.getData().forEach(i -> {
            if (Strings.CS.startsWith(i.getKey(), prefix)) {
                mapper.put(Strings.CS.replace(i.getKey(), prefix, StringUtils.EMPTY), get.apply(i));
            }
        });
        BiConsumer<Map<String, String>, BiConsumer<I18nName, String>> route = params.getRoute();
        if (route != null) {
            route.accept(mapper, i18nPair(params.getI18nTp()).getConsName());
        }
    }

    /**
     * 读取vue的配置文件 转为语言对应的数据
     * @param path 路径
     * @return 值
     */
    private static Map<I18nTp, Map<String, Object>> i18nValueMap (String path){
        File file = new File(path);
        isVue(file, path);
        StringBuilder sb;
        Map<I18nTp, StringBuilder> outMap = new HashMap<>();
        for (Map.Entry<String, String> entry : MAP.entrySet()) {
            String tp = StringUtils.substringBetween(entry.getKey(), BaseConstants.UNDERLINE, BaseConstants.SPOT);
            I18nTp itp = I18nTp.ofCode(tp);
            sb = outMap.get(itp);
            if (sb == null) {
                sb = new StringBuilder();
            }
            sb.append(entry.getValue()).append(",\n");
            outMap.put(itp, sb);
        }
        Map<I18nTp, Map<String, Object>> valuesMap = new HashMap<>();
        for (Map.Entry<I18nTp, StringBuilder> e : outMap.entrySet()) {
            valuesMap.put(e.getKey(), appendColon(e.getValue()));
        }
        return valuesMap;
    }


    /**
     * 获取到 js格式的json
     * @param obj 对象
     * @return json
     */
    private static String toJsJson(Object obj){
        String json = JsonUtil.printerJson(obj);
        StringBuilder result = new StringBuilder();
        Matcher matcher = PATTERN.matcher(json);
        while (matcher.find()) {
            String group = matcher.group();
            group = group.replaceAll("\"", StringUtils.EMPTY);
            matcher.appendReplacement(result, group);
        }
        matcher.appendTail(result);
        return result.toString();
    }

    /**
     * 判断map是不是 子模块
     * @param v 值
     * @return 是否子模块
     */
    private static boolean firstValueIsMap(Object v){
        Map<String, Object> map = PojoUtil.cast(v);
        Object firstValue = map.values().stream().findFirst().orElse(null);
        return !(firstValue instanceof String);
    }


    /**
     * 以逗号分割 写值到map对象 平铺
     * @param ii 国际化值
     * @param value 对应的语言
     * @param map map
     */
    private static void setMap(I18nInfo ii, Function<I18nInfo, String> value, Map<String, Object> map) {
        String key = ii.getKey();
        String[] ks = StringUtils.split(key, BaseConstants.SPOT);
        int len = ks.length;
        Map<String, Object> pmp = map;
        Map<String, Object> tmp = null;
        for (int i = 0, j = len - 1; i < j; i++) {
            tmp = PojoUtil.cast(pmp.get(ks[i]));
            if (tmp == null) {
                tmp = new HashMap<>();
            }
            pmp.put(ks[i], tmp);
            pmp = tmp;
        }
        if (tmp != null) {
            tmp.put(ks[len - 1], value.apply(ii));
        }
    }


    /**
     * 判断是否为vue
     * @param file 文件
     * @param rootPath 根目录
     */
    private static void isVue(File file, String rootPath){
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            assert files != null;
            for (File f : files) {
                isVue(f, rootPath);
            }
        } else {
            String fm = file.getName();
            if (fm.length() == 5) {
                vueJson(file, rootPath, fm);
            }
        }
    }
    

    /**
     * 将map转为国际化数据
     * @param map map对象
     * @param key key
     * @param bc 语言 对应的set函数
     * @param result 结果
     */
    private static void toMap (Map<String, Object> map, String key, BiConsumer<I18nInfo, String> bc, Map<String, I18nInfo> result){
        Object v;
        String k;
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            v = entry.getValue();
            k = key;
            if (StringUtils.isNotBlank(key)) {
                k += BaseConstants.SPOT;
            }
            k += entry.getKey();
            if (v instanceof String s) {
                I18nInfo name = result.get(k);
                if (name == null) {
                    name = new I18nInfo();
                    name.setKey(k);
                }
                bc.accept(name, s);
                result.put(k, name);
            } else if (v instanceof Map<?, ?> m) {
                toMap(PojoUtil.cast(m), k, bc, result);
            }
        }
    }

    /**
     * 获取到java 的国际化配置
     * 转为 验证 和异常 的map对象
     * @param map 解析出来的
     * @param sort 排序
     * @return 最后结果
     */
    private static Map<String, List<I18nInfo>> readJavaI18n (Map<String, List<String>> map, boolean sort){
        Map<String, Map<String, I18nInfo>> config = new HashMap<>();
        String key;
        for (Map.Entry<String, List<String>> e : map.entrySet()) {
            String[] ns = StringUtils.split(e.getKey(), BaseConstants.UNDERLINE);
            I18nTp tp = I18nTp.ofCode(ns[1]);
            BiConsumer<I18nInfo, String> bc = setFun(tp);
            Map<String, I18nInfo> temp = config.get(ns[0]);
            if (temp == null) {
                temp = new HashMap<>();
            }
            for (String v : e.getValue()) {
                if (StringUtils.isNotBlank(v) && Strings.CS.contains(v, EQ)) {
                    String[] vs = StringUtils.split(v, EQ);
                    key = vs[0].trim();
                    I18nInfo info = temp.get(key);
                    if (info == null) {
                        info = new I18nInfo();
                        info.setKey(key);
                    }
                    String cv = vs[1];
                    if (StringUtils.isNotBlank(cv) && (tp == I18nTp.CN || tp == I18nTp.ES)) {
                        cv = StringUtils.trim(cv);
                        cv = StringEscapeUtils.unescapeJava(cv);
                    }
                    bc.accept(info, cv);
                    temp.put(key, info);
                }
            }
            config.put(ns[0], temp);
        }
        Map<String, List<I18nInfo>> m = new HashMap<>();
        for (Map.Entry<String, Map<String, I18nInfo>> e : config.entrySet()) {
            List<I18nInfo> list = new ArrayList<>();
            e.getValue().forEach((k, v) -> list.add(v));
            if (sort) {
                list.sort(Comparator.comparing(I18nInfo::getKey));
            }
            m.put(e.getKey(), list);
        }
        return m;
    }

    /**
     * 获取到java 的国际化配置
     * @return 国际化类型_语言 所有的数据
     */
    private static Map<String, List<String>> readJavaI18n(){
        ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        Map<String, List<String>> map = new HashMap<>();
        try {
            Resource[] resources = resolver.getResources("classpath*:i18n/**/*.*");
            for (Resource resource : resources) {
                String path = resource.getFilename();
                List<String> list = IOUtils.readLines(resource.getInputStream(), Charset.defaultCharset());
                String key = key(path);
                List<String> temp = map.get(key);
                if (temp == null) {
                    temp = new ArrayList<>();
                }
                temp.addAll(list);
                map.put(key, temp);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return map;
    }

    /**
     * 根据文件名称 组成key
     * @param name 文件名称
     * @return key
     */
    private static String key (String name){
        String[] ns = StringUtils.split(name, BaseConstants.UNDERLINE);
        int last = ns.length - 2;
        String p = MSG;
        if (Strings.CS.equals(ns[last - 1], VALIDATOR)) {
            p = VALIDATOR;
        } else if (Strings.CS.equals(ns[last - 1], TITLE)) {
            p = TITLE;
        }
        return p + BaseConstants.UNDERLINE + ns[last];
    }

    /**
     * 将  vue配置的 国际化文件内容 转为 标准的json
     * 最后转为map
     * @param sb 数据
     * @return  map
     */
    private static Map<String, Object> appendColon(StringBuilder sb){
        String input = "{\n" + sb + "\n}";
        String[] lines = input.split("\n");
        List<String> list = new ArrayList<>();
        for (String line : lines) {
            if (Strings.CS.contains(line, BaseConstants.COLON)) {
                String[] arays = StringUtils.split(line, BaseConstants.COLON);
                try {
                    list.add("\""  + arays[0].trim() + "\"" + BaseConstants.COLON + arays[1]);
                } catch (Exception e) {
                    log.error(e.getMessage());
                }

            } else {
                list.add(line);
            }
        }
        String json = String.join("\n", list);
        // 去除所有换行
        json = json.replaceAll("\r", StringUtils.EMPTY).replaceAll("\n", StringUtils.EMPTY);
        StringBuilder result = new StringBuilder();
        Pattern pattern = Pattern.compile(",\\s+}");
        Matcher matcher = pattern.matcher(json);
        while (matcher.find()) {
            String group = matcher.group();
            group = group.replaceAll("\\s+", StringUtils.EMPTY);
            matcher.appendReplacement(result, group);
        }
        matcher.appendTail(result);
        String newJson = result.toString();
        newJson = newJson.replaceAll(",}", "}");
        try {
            return JsonUtil.toMap(newJson);
        } catch (Exception e) {
            log.error("错误的json：{}", newJson);
            throw e;
        }

    }

    /**
     * 读取国际化vue的js 文件 根目录的 取第二个 {}内的数据
     * 其他的取第一层 {}，再在外层加上 模块
     * @param file 文件
     * @param rootPath 根目录
     * @param fm 文件名称
     */
    private static void vueJson(File file, String rootPath, String fm){
        String path = file.getParent().replace(rootPath, StringUtils.EMPTY);
        String name = BaseConstants.AT;
        String text = null;
        try {
            text = FileUtils.readFileToString(file, Charset.defaultCharset());
            if (StringUtils.isNotBlank(path)) {
                text = StringUtils.substringAfter(text, "{");
                text = StringUtils.substringBeforeLast(text, "}");
                name = path.replace(File.separator, StringUtils.EMPTY);
                text = name + ": {\n" + text + "\n}";
            } else {
                text = StringUtils.substringAfter(text, "{");
                text = StringUtils.substringBeforeLast(text, "}");
                text = StringUtils.substringBeforeLast(text, "}");
                text = text + "\n}\n";
            }
        } catch (Exception e) {
            log.error(e.getMessage());
        }
        MAP.put(name + BaseConstants.UNDERLINE + fm, text);
    }

    /**
     * 将新值设置到map中
     * @param structure 原有的结构map
     * @param vm 新值map
     */
    private static void recursionSet (Map<String, Object> structure, Map<String, Object> vm){
        for (Map.Entry<String, Object> e : structure.entrySet()) {
            Object v = e.getValue();
            if (v instanceof String) {
                e.setValue(vm.get(e.getKey()));
            } else {
                Map<String, Object> vnv = PojoUtil.cast(vm.get(e.getKey()));
                recursionSet(PojoUtil.cast(v), vnv);
            }
        }
    }

    /**
     * 根据类型获取到 get
     * @param tp 语言
     * @return 函数
     */
    private static Function<I18nInfo, String> getFun(I18nTp tp){
        return i18nPair(tp).getFunInfo();
    }

    /**
     * 获取到set的函数
     * @param tp 语言
     * @return 函数
     */
    private static BiConsumer<I18nInfo, String> setFun(I18nTp tp){
        return i18nPair(tp).getConsInfo();
    }

    private static I18nSetGet i18nPair (I18nTp tp){
        return MapUtils.getObject(MAPPER, tp, MAPPER.get(I18nTp.CN));
    }
}
