package cn.sinozg.applet.common.interceptor;

import cn.sinozg.applet.common.annotation.RepeatSubmit;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.filter.WrapperRequest;
import cn.sinozg.applet.common.holder.UserContextHolder;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.service.FrameworkAuthService;
import cn.sinozg.applet.common.utils.CypherUtil;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.MsgUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.common.utils.RedisUtil;
import cn.sinozg.applet.common.utils.TirsciUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.ModelAndView;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
* 防止重复提交 参数签名拦截器
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 10:10
**/
@Component
public class ParamsInterceptor implements HandlerInterceptor {
    @Resource
    private AppValue app;

    @Resource
    private FrameworkAuthService tokenService;

    private final Logger log = LoggerFactory.getLogger(ParamsInterceptor.class);

    @Override
    public boolean preHandle(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler) throws Exception {
        // 设置多线程数据
        if(HttpMethod.OPTIONS.matches(request.getMethod())) {
            response.setStatus(HttpStatus.NO_CONTENT.value());
            return false;
        }
        UserContextHolder.setInfo(request);
        if (!(handler instanceof HandlerMethod handlerMethod)) {
            return true;
        }
        String params = StringUtils.EMPTY;
        if (request instanceof WrapperRequest) {
            params = WebUtil.getBodyString(request);
        }
        // body参数为空，获取Parameter的数据 对get请求
        if (StringUtils.isEmpty(params) && HttpMethod.GET.name().equals(request.getMethod())){
            params = getRequestParams(request);
            log.info("get 请求 的参数为 {}", params);
        }
        if (StringUtils.isBlank(params)) {
            params = BaseConstants.EMPTY_JSON;
        }
        Method method = handlerMethod.getMethod();
        // 参数签名校验
        if (app.getSign().isSignEnable() && !this.signParams(request, response, params)) {
            return false;
        }
        // 重复提交校验
        RepeatSubmit rs = method.getAnnotation(RepeatSubmit.class);
        if (rs != null && this.isRepeatSubmit(request, rs, params)) {
            httpError(request, response, "BIZ000100016");
            return false;
        }
        return true;
    }

    @Override
    public void postHandle(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler, ModelAndView modelAndView) throws Exception {
        HandlerInterceptor.super.postHandle(request, response, handler, modelAndView);
    }
    @Override
    public void afterCompletion(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler, Exception ex) throws Exception {
        UserContextHolder.clear();
    }
    /**
     * 获取到 get请求 参数 进行签名
     * @param request 请求参数
     * @return get请求的参数
     */
    private String getRequestParams (HttpServletRequest request){
        Map<String, String> urlPathMap = PojoUtil.cast(request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE));
        Map<String, String[]> paramsMap = request.getParameterMap();
        String[] arrays;
        Map<String, String[]> jsonMap = new TreeMap<>(Comparator.naturalOrder());
        for (Map.Entry<String, String[]> e : paramsMap.entrySet()) {
            arrays = e.getValue();
            Arrays.sort(arrays);
            jsonMap.put(e.getKey(), arrays);
        }
        if (MapUtils.isNotEmpty(urlPathMap)) {
            List<String> list = new ArrayList<>();
            urlPathMap.forEach((k, v) -> list.add(v));
            list.sort(String::compareTo);
            jsonMap.put("get_url_path_param", PojoUtil.toArray(list, String.class));
        }
        return JsonUtil.toJson(jsonMap);
    }

    /**
     * 参数签名校验
     * @param request 请求request
     * @param response 返回
     * @param params 请求参数
     * @return 签名是否正确
     */
    private boolean signParams (HttpServletRequest request, HttpServletResponse response, String params){
        String nonce = request.getHeader(HeaderConstants.X_NONCE);
        String signature = request.getHeader(HeaderConstants.X_SIGNATURE);
        String time = StringUtils.trimToEmpty(request.getHeader(HeaderConstants.X_TIMESTAMP));
        // 验证请求头是否存在
        if(StringUtils.isAnyBlank(nonce, signature, time)){
            httpError(request, response, "BIZ000100019");
            return false;
        }
        // 重放验证 判断timestamp时间戳与当前时间是否操过60s（过期时间根据业务情况设置）,如果超过了就提示签名过期。
        long now = System.currentTimeMillis() / 1000;
        int signMaxTime = app.getSign().getSignExpTime();
        Long timestamp = TirsciUtil.tryCatch(() -> Long.parseLong(time));
        if (timestamp == null || Math.abs(now - timestamp) > signMaxTime) {
            httpError(request, response, "BIZ000100020");
            return false;
        }
        // 2. 判断nonce
        String redisKey = String.format(BaseRedisKeys.PARAMS_NONE_KEY, nonce);
        boolean exists = RedisUtil.setIfAbsentObject(redisKey, nonce, Duration.ofSeconds(signMaxTime));
        if(!exists){
            //请求重复
            httpError(request, response, "BIZ000100021");
            return false;
        }
        // 检查签名是否正确
        boolean accept = CypherUtil.signature(true, signature, nonce, time, params);
        if (!accept) {
            httpError(request, response, "BIZ000100022");
            return false;
        }
        return true;
    }


    /**
     * 验证是否重复提交由子类实现具体的防重复提交的规则
     * @param request 请求request
     * @param rs 自定义注解
     * @param json 请求参数
     * @return 是否重复提交
     */
    private boolean isRepeatSubmit(HttpServletRequest request, RepeatSubmit rs, String json) {
        String params = BaseConstants.EMPTY_JSON;
        if (rs.withParams()) {
            params = json;
        }
        String paramsMd5 = CypherUtil.md5(params);
        // 请求地址（作为存放cache的key值）
        String url = request.getRequestURI();
        // 唯一值（没有消息头则使用请求地址）同一用户
        String submitKey = tokenService.token();
        if (StringUtils.isEmpty(submitKey)) {
            submitKey = url;
        }
        // 唯一标识（指定key + url + MD5(请求消息)）
        String cacheRepeatKey = String.format(BaseRedisKeys.REPEAT_SUBMIT_KEY, submitKey, paramsMd5);
        long time = RedisUtil.getIncr(cacheRepeatKey, Duration.ofSeconds(10));
        return time >= rs.times();
    }

    private void httpError (HttpServletRequest request, HttpServletResponse response, String code){
        MsgUtil.httpError(response, request, null, HttpStatus.OK, code, (Object) null);
    }

}
