/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.config;

import cn.sinozg.applet.common.properties.AppValue;
import io.swagger.v3.core.util.PrimitiveType;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.ExternalDocumentation;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.info.License;
import io.swagger.v3.oas.models.parameters.Parameter;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import jakarta.annotation.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springdoc.core.customizers.OperationCustomizer;
import org.springdoc.core.models.GroupedOpenApi;
import org.springdoc.core.utils.Constants;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * <a href="http://127.0.0.1:8081/doc.html">swagger地址</a>
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-23 下午 06:53
 **/
@Configuration
@ConditionalOnProperty(name = Constants.SPRINGDOC_ENABLED, matchIfMissing = true)
public class SwaggerConfig {

    @Resource
    private AppValue app;

    private final Logger log = LoggerFactory.getLogger(SwaggerConfig.class);

    private static final String URL = "https://www.jiubankeji.cn";

    @Bean
    public OperationCustomizer customize() {
        // add LocalTime
        PrimitiveType.enablePartialTime();
        return (o, h) -> o.addParametersItem(
                new Parameter()
                        .in(ParameterIn.HEADER.toString())
                        .required(false)
                        .description("用户id")
                        .name("UserId"));
    }

    @Bean
    public GroupedOpenApi publicApi() {
        return GroupedOpenApi.builder()
                .group("SwaggerAPI")
                .pathsToMatch("/**")
                .build();
    }

    @Bean
    public OpenAPI springShopOpenApi() {
        String securitySchemeName = "Authorization OAuth2";
        return new OpenAPI()
                .info(new Info().title("项目接口文档")
                        .description(app.getDescription() +  " 接口信息")
                        .version(app.getVersion())
                        .contact(new Contact().name("Wuhan Jubilant Cloud Technology Co., Ltd.").email("xie.y.bin@163.com").url(URL))
                        .license(new License().name("Apache 2.0").url("https://springdoc.org/")))
                .externalDocs(new ExternalDocumentation()
                        .description("Swagger Api Documentation")
                        .url(URL))
                .addSecurityItem(new SecurityRequirement().addList(securitySchemeName))
                .components(new Components()
                        .addSecuritySchemes(securitySchemeName, new SecurityScheme()
                                .name(securitySchemeName)
                                .type(SecurityScheme.Type.APIKEY)
                                .in(SecurityScheme.In.HEADER)
                                .description("OAuth2 token flow.")
                        )

                );
    }
}
