/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.filter;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.holder.MdcContextHolder;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.service.CipherService;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;

import java.io.IOException;


/**
 * 包装http 请求 过滤器
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 09:41
 **/
public class WrapperFilter implements Filter {

    /** 是否启用防重复 */
    private AppValue app;

    private CipherService cipherService;

    private final Logger log = LoggerFactory.getLogger(WrapperFilter.class);

    public WrapperFilter(){
    }

    public WrapperFilter(AppValue app, CipherService cipherService){
        this.app = app;
        this.cipherService = cipherService;
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        ServletRequest servletRequest = null;
        try {
            if (request instanceof HttpServletRequest hsr) {
                // 链路信息
                String traceId = hsr.getHeader(HeaderConstants.X_TRACE_ID);
                MdcContextHolder.put(traceId);
                boolean isWrapper = app.getApiUrl().isWrapperEnable();
                // 包装 http请求
                if (isWrapper) {
                    String contentType = StringUtils.substringBefore(request.getContentType(), ";");
                    isWrapper = StringUtils.equalsAnyIgnoreCase(contentType, MediaType.APPLICATION_JSON_VALUE, BaseConstants.APPLICATION_JSON);
                }
                if (isWrapper) {
                    try {
                        servletRequest = new WrapperRequest(hsr, response, app.getSign(), cipherService);
                    } catch (Exception e) {
                        log.error("包装请求参数失败！", e);
                        PojoWebUtil.resolveException(hsr, (HttpServletResponse) response, new RuntimeException("Decryption parameter exception!"));
                        return;
                    }
                }
            }
            if (servletRequest == null) {
                servletRequest = request;
            }
            chain.doFilter(servletRequest, response);
        } finally {
            MdcContextHolder.clear();
        }
    }
}
