/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.interceptor;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.base.BasePageResponse;
import cn.sinozg.applet.common.core.base.BaseResponse;
import cn.sinozg.applet.common.holder.MdcContextHolder;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.service.CipherService;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import java.util.Objects;


/**
 * 响应报文赋值
 * @Author: xyb
 * @Description: 
 * @Date: 2023-04-21 下午 05:42
 **/
@Order(2)
@ControllerAdvice
public class ResponseBodyAdviceAdapter implements ResponseBodyAdvice<Object> {

	@Resource
	private AppValue app;

	@Resource
	private CipherService cipherService;

	private final Logger log = LoggerFactory.getLogger(ResponseBodyAdviceAdapter.class);

	@Override
	public Object beforeBodyWrite(Object o, @NotNull MethodParameter mp, @NotNull MediaType t, @NotNull Class<? extends HttpMessageConverter<?>> c,
								  @NotNull ServerHttpRequest q, @NotNull ServerHttpResponse p) {
		BaseResponse<?> response = (BaseResponse<?>) o;
		if (StringUtils.isBlank(response.getBizStatus())) {
			// 设置响应 状态
			response.setBizStatus(response.getErrorInfo() == null ? BaseConstants.SUCCESS : BaseConstants.FAIL);
		}
		response.setTraceId(MdcContextHolder.get());
		if (app.getSign().isCryptoEnable()) {
			ServletServerHttpRequest httpRequest = (ServletServerHttpRequest) q;
			HttpServletRequest request = httpRequest.getServletRequest();
			try {
				return cipherService.encryptJson(request, response);
			} catch (Exception e) {
				log.error("接口【{}】，返回数据进行加密异常，{}！", Objects.requireNonNull(mp.getMethod()).getName(), e.getMessage());
			}
		}
		return o;
	}

	/**
	 * 校验拦截器是否处理：仅处理BaseResponse对象
	 * @param p MethodParameter
	 * @param c c
	 * @return 是否支持
	 */
	@Override
	public boolean supports(@NotNull MethodParameter p, @NotNull Class<? extends HttpMessageConverter<?>> c) {
		boolean flag = false;
		try {
			Class<?> cl = p.getParameterType();
			// 如果返回对象是BaseResponse的子类，则处理(是否包含标准数据翻译注解不考虑，避免重复处理)
			flag = cl.isAssignableFrom(BaseResponse.class) || cl.isAssignableFrom(BasePageResponse.class)
				|| cl.isAssignableFrom(void.class);
		} catch (Exception e) {
			log.error("返回参数拦截器获取返回参数类型错误，", e);
		}
		return flag;
	}

}
