/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.filter;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.core.model.RequestParameterDecrypt;
import cn.sinozg.applet.common.properties.SignValue;
import cn.sinozg.applet.common.service.CipherService;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import jakarta.servlet.ServletInputStream;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 构建可重复读取inputStream的request
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 09:41
 **/
public class WrapperRequest extends HttpServletRequestWrapper {

    private final byte[] body;

    private final SignValue sign;

    private final CipherService cipherService;

    private final Logger log = LoggerFactory.getLogger(WrapperRequest.class);

    private final Map<String, String> headerMap = new HashMap<>();

    public WrapperRequest(HttpServletRequest request, ServletResponse response, SignValue sign, CipherService cipherService) throws IOException {
        super(request);
        this.sign = sign;
        this.cipherService = cipherService;
        request.setCharacterEncoding(BaseConstants.UTF8);
        response.setCharacterEncoding(BaseConstants.UTF8);
        body = decrypt(request);
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream()));
    }

    @Override
    public ServletInputStream getInputStream() {
        return new WrapperInputStream(body);
    }

    @Override
    public String getHeader(String name) {
        String value = super.getHeader(name);
        if (headerMap.containsKey(name)) {
            value = headerMap.get(name);
        }
        return value;
    }

    @Override
    public Enumeration<String> getHeaderNames() {
        List<String> names = Collections.list(super.getHeaderNames());
        names.addAll(headerMap.keySet());
        return Collections.enumeration(names);
    }

    @Override
    public Enumeration<String> getHeaders(String name) {
        List<String> values = Collections.list(super.getHeaders(name));
        if (headerMap.containsKey(name)) {
            values.add(headerMap.get(name));
        }
        return Collections.enumeration(values);
    }

    /**
     * 解密请求参数
     * @param request request
     * @return 解密后的数据
     * @throws IOException 异常
     */
    private byte[] decrypt(HttpServletRequest request) throws IOException {
        String jsonBody = WebUtil.getBodyString(this.getRequest());
        byte[] bytes = null;
        if (PojoWebUtil.nonEncrypt(sign, request)) {
            RequestParameterDecrypt decrypt = JsonUtil.toPojo(jsonBody, RequestParameterDecrypt.class);
            if (decrypt == null || StringUtils.isAnyEmpty(decrypt.getData(), decrypt.getPublicKey(), decrypt.getSymmetricEncrypt())) {
                throw new IOException("Encryption parameter format error!");
            }
            headerMap.put(HeaderConstants.X_PUB_KEY, decrypt.getPublicKey());
            try {
                bytes = cipherService.decryptJson(decrypt.getData(), sign.getPrivateKey(), decrypt.getSymmetricEncrypt());
            } catch (Exception e) {
                log.error("解密参数错误！", e);
            }
            if (bytes == null) {
                throw new IOException("Decryption parameter error!");
            }
        } else {
            bytes = jsonBody.getBytes(StandardCharsets.UTF_8);
        }
        return bytes;
    }
}
