/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.biz.framework.cleanup;

import cn.sinozg.applet.common.function.FunctionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

/**
 * 用于清理日志表
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-21 23:43
 */
@Component
public class LogCleanupAsync {

    private static final Logger log = LoggerFactory.getLogger(LogCleanupAsync.class);

    /**
     * 用于清理日志，异步，批量处理
     *
     * @param name 任务名称
     * @param maxRetries 最大重试次数
     * @param params 参数
     * @param del 删除逻辑，需加事物，不要在这层使用事物太大
     * @param <T> 参数类型
     */
    @Async
    public <T> void logCleanup(String name, int maxRetries, T params, FunctionException<T, Long> del){
        long count = 1;
        int retryCount = 0;
        while (count > 0) {
            try {
                count = del.apply(params);
                // 重试成功，重置重试计数
                retryCount = 0;
                TimeUnit.MILLISECONDS.sleep(100);
            } catch (Exception e) {
                log.error("删除{}日志失败: {}，错误: {}", name, e.getMessage(), e);
                retryCount++;
                if (retryCount >= maxRetries) {
                    log.error("{} 最大重试次数到达，停止删除日志", name);
                    // 达到最大重试次数，退出循环
                    break;
                }
                try {
                    TimeUnit.MILLISECONDS.sleep(2000);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }
}
