/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.biz.system.service.impl;

import cn.sinozg.applet.biz.framework.service.FrameworkUserLoginService;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.core.model.LoginUserVo;
import cn.sinozg.applet.common.core.model.UserThreadInfo;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.holder.UserContextHolder;
import cn.sinozg.applet.common.service.AuthStorageService;
import cn.sinozg.applet.common.service.FrameworkAuthService;
import cn.sinozg.applet.common.service.FrameworkUserService;
import jakarta.annotation.Resource;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * 系统登录用户信息 实现
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:20
 **/
@Service
public class FrameworkUserServiceImpl implements FrameworkUserService {

    @Resource
    private FrameworkUserLoginService service;

    @Resource
    private FrameworkAuthService frameworkAuthService;

    @Resource
    private AuthStorageService authStorageService;

    private final Logger log = LoggerFactory.getLogger(FrameworkUserServiceImpl.class);

    @Override
    public LoginUserVo refreshUserInfo() {
        return userInfo(true);
    }

    @Override
    public LoginUserVo userInfo() {
        return userInfo(false);
    }

    @Override
    public LoginUserVo setUserInfo (String token, String uid, LoginUserVo user){
        if (user == null) {
            user = queryDb(uid);
        }
        String key = redisKey(token, uid);
        long timeout = frameworkAuthService.tokenTimeOut();
        authStorageService.setCacheObject(key, user, timeout);
        return user;
    }

    @Override
    public LoginUserVo getUserInfo(String token, String uid) {
        if (!StringUtils.isAllBlank(token, uid)) {
            String key = redisKey(token, uid);
            return authStorageService.getCacheObject(key);
        }
        return null;
    }

    /**
     * 获取用户信息
     * 先从 缓存获取 再从数据库获取
     * @param force 是否强制更新
     * @return 用户信息
     */
    private LoginUserVo userInfo(boolean force){
        String token = frameworkAuthService.token();
        // 子线程下 或错误
        String uid = frameworkAuthService.loginId();
        if (StringUtils.isBlank(token)) {
            UserThreadInfo info = UserContextHolder.getInfo();
            if (info != null) {
                uid = info.getUid();
                token = info.getToken();
            }
        }
        LoginUserVo user = null;
        if (!StringUtils.isAllBlank(token, uid)) {
            String key = redisKey(token, uid);
            if (!force) {
                // redis 查询
                user = authStorageService.getCacheObject(key);
            }
            // 数据库获取
            if (user == null && StringUtils.isNotBlank(uid)) {
                user = queryDb(uid);
                // 存储
                setUserInfo(token, uid, user);
            }
        }
        return user;
    }

    /**
     * 从数据库获取用户信息
     * @param uid 用户id
     * @return 用户信息
     */
    private LoginUserVo queryDb (String uid){
        // 用户基本信息
        LoginUserVo user = service.userInfo(uid);
        if (user == null) {
            throw new CavException("BIZ000100002");
        }
        // 权限查询
        return user;
    }

    /**
     * 用户信息 redis 里面的key值
     * @param token token
     * @param uid 用户id
     * @return redis的key
     */
    private String redisKey (String token, String uid){
        return String.format(BaseRedisKeys.USER_INFO, token, uid);
    }
}
