/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.config;

import cn.sinozg.applet.common.filter.WrapperFilter;
import cn.sinozg.applet.common.interceptor.ParamsInterceptor;
import cn.sinozg.applet.common.properties.ApiUrlValue;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.service.CipherService;
import cn.sinozg.applet.common.service.FrameworkAuthService;
import cn.sinozg.applet.common.service.FrameworkInterceptorService;
import cn.sinozg.applet.common.service.FrameworkTenantService;
import cn.sinozg.applet.common.utils.SpringUtil;
import jakarta.annotation.Resource;
import org.apache.commons.collections4.CollectionUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 鉴权 全局拦截器
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 10:09
 **/
@Configuration
public class SystemWebConfig implements WebMvcConfigurer {

    @Resource
    private AppValue app;

    @Resource
    private CipherService cipherService;

    @Resource
    private ParamsInterceptor paramsInterceptor;

    @Resource
    private FrameworkAuthService authService;

    @Resource
    private FrameworkTenantService tenantService;

    private final Logger log = LoggerFactory.getLogger(SystemWebConfig.class);

    @Override
    public void addInterceptors(@NotNull InterceptorRegistry registry) {
        log.info("加载系统 拦截器 ...... ");
        ApiUrlValue apiUrl = app.getApiUrl();
        List<FrameworkInterceptorService> lastOrder = new ArrayList<>();
        // 因为使用的是function，在没有实现类的时候，使用注入的方式会报错，所以此处使用静态获取
        Map<String, FrameworkInterceptorService> interceptorServiceMap = SpringUtil.beansOfType(FrameworkInterceptorService.class);
        for (Map.Entry<String, FrameworkInterceptorService> entry : interceptorServiceMap.entrySet()) {
            FrameworkInterceptorService interceptor = entry.getValue();
            Order order = interceptor.getClass().getAnnotation(Order.class);
            if (order != null && order.value() < 100) {
                interceptor.addInterceptor(registry);
            } else {
                lastOrder.add(interceptor);
            }
        }
        // 参数校验
        if (CollectionUtils.isNotEmpty(apiUrl.getParamsPatterns())) {
            registry.addInterceptor(paramsInterceptor).addPathPatterns(apiUrl.getParamsPatterns());
        }
        // token 拦截
        HandlerInterceptor authInterceptor = authService.authInterceptor();
        if (authInterceptor != null) {
            registry.addInterceptor(authInterceptor)
                    .addPathPatterns(apiUrl.getAuthPatterns())
                    .excludePathPatterns(apiUrl.getAuthExcludePath());
        }
        tenantService.expiredCheckInterceptor(registry);
        if (CollectionUtils.isNotEmpty(lastOrder)) {
            lastOrder.forEach(i -> i.addInterceptor(registry));
        }
    }

    /**
     * http请求 包装
     * @return 过滤器
     */
    @Bean
    public FilterRegistrationBean<WrapperFilter> registrationBean (){
        ApiUrlValue apiUrl = app.getApiUrl();
        FilterRegistrationBean<WrapperFilter> bean = new FilterRegistrationBean<>();
        bean.setFilter(new WrapperFilter(app, cipherService));
        bean.addUrlPatterns(apiUrl.getWrapperUrl());
        return bean;
    }
}
