/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.handler;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.base.BaseResponse;
import cn.sinozg.applet.common.core.model.ErrorMsgInfo;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.utils.MsgUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.common.utils.VerifyUtil;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import java.util.Set;

/**
 * 全局异常处理器
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:10
 **/
@RestControllerAdvice
public class GlobalExceptionHandler {

    @Resource
    private AppValue app;

    private static final Logger log = LoggerFactory.getLogger(GlobalExceptionHandler.class);

    /**
     * 在controller里面内容执行之前，
     * 校验一些参数不匹配啊，
     * Get post方法不对啊之类的
     * @param request HttpServletRequest
     * @param e 异常
     * @return 结果集
     */
    @ResponseBody
    @ExceptionHandler(Exception.class)
    public BaseResponse<String> jsonHandler(HttpServletRequest request, Exception e) {
        log.error("GlobalExceptionHandler.jsonHandler start");
        ErrorMsgInfo.Builder builder = new ErrorMsgInfo.Builder(BaseConstants.GLOBAL_EXCEPTION_MESSAGE, e, request);
        if (app.getDebug() != null && app.getDebug()) {
            builder.debug();
        }
        return MsgUtil.error(builder);
    }
    /**
     * 业务异常处理
     * @param request HttpServletRequest
     * @param e 异常
     * @return 结果集
     */
    @ResponseBody
    @ExceptionHandler(CavException.class)
    public BaseResponse<String> cavException(HttpServletRequest request, CavException e) {
        log.error("business exception start");
        return MsgUtil.error(new ErrorMsgInfo.Builder(e.getCode(), e, request, e.getParameters()));

    }

    /**
     * 自定义校验
     * @param request 参数
     * @param e 异常
     * @return 错误
     */
    @ResponseBody
    @ExceptionHandler(ConstraintViolationException.class)
    public BaseResponse<String> constraintViolationException(HttpServletRequest request, ConstraintViolationException e) {
        log.error("constraintViolationException start");
        Set<ConstraintViolation<?>> violations = e.getConstraintViolations();
        String msg = VerifyUtil.validateErrorInfo(PojoUtil.cast(violations));
        return MsgUtil.error(new ErrorMsgInfo.Builder("BIZ000100014", e, request, msg));
    }

    /**
     * 请求参数异常处理
     * @param request HttpServletRequest
     * @param e 异常
     * @return 结果集
     */
    @ResponseBody
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResponse<String> argumentNotValidException(HttpServletRequest request, MethodArgumentNotValidException e) {
        log.error("MethodArgumentNotValidException start");
        String msg = VerifyUtil.errorInfo(e.getBindingResult());
        return MsgUtil.error(new ErrorMsgInfo.Builder("BIZ000100014", e, request, msg));
    }
}
