/*
 * Decompiled with CFR 0.152.
 */
package org.agrona;

import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.function.BiConsumer;
import org.agrona.BufferUtil;
import org.agrona.ErrorHandler;
import org.agrona.LangUtil;
import org.agrona.UnsafeAccess;

public final class IoUtil {
    public static final int BLOCK_SIZE = 4096;
    private static final byte[] FILLER = new byte[4096];
    private static final int MAP_READ_ONLY = 0;
    private static final int MAP_READ_WRITE = 1;
    private static final int MAP_PRIVATE = 2;

    private IoUtil() {
    }

    public static void fill(FileChannel fileChannel, long position, long length, byte value) {
        try {
            byte[] filler;
            if (0 != value) {
                filler = new byte[4096];
                Arrays.fill(filler, value);
            } else {
                filler = FILLER;
            }
            ByteBuffer byteBuffer = ByteBuffer.wrap(filler);
            fileChannel.position(position);
            int blocks = (int)(length / 4096L);
            int blockRemainder = (int)(length % 4096L);
            for (int i = 0; i < blocks; ++i) {
                byteBuffer.position(0);
                fileChannel.write(byteBuffer);
            }
            if (blockRemainder > 0) {
                byteBuffer.position(0);
                byteBuffer.limit(blockRemainder);
                fileChannel.write(byteBuffer);
            }
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    public static void delete(File file, boolean ignoreFailures) {
        if (file.exists()) {
            File[] files;
            if (file.isDirectory() && null != (files = file.listFiles())) {
                for (File f : files) {
                    IoUtil.delete(f, ignoreFailures);
                }
            }
            if (!file.delete() && !ignoreFailures) {
                try {
                    Files.delete(file.toPath());
                }
                catch (IOException ex) {
                    LangUtil.rethrowUnchecked(ex);
                }
            }
        }
    }

    public static void delete(File file, ErrorHandler errorHandler) {
        try {
            if (file.exists()) {
                File[] files;
                if (file.isDirectory() && null != (files = file.listFiles())) {
                    for (File f : files) {
                        IoUtil.delete(f, errorHandler);
                    }
                }
                if (!file.delete()) {
                    Files.delete(file.toPath());
                }
            }
        }
        catch (Exception ex) {
            errorHandler.onError(ex);
        }
    }

    public static void ensureDirectoryExists(File directory, String descriptionLabel) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new IllegalArgumentException("could not create " + descriptionLabel + " directory: " + directory);
        }
    }

    public static void ensureDirectoryIsRecreated(File directory, String descriptionLabel, BiConsumer<String, String> callback) {
        if (directory.exists()) {
            IoUtil.delete(directory, false);
            callback.accept(directory.getAbsolutePath(), descriptionLabel);
        }
        if (!directory.mkdirs()) {
            throw new IllegalArgumentException("could not create " + descriptionLabel + " directory: " + directory);
        }
    }

    public static void deleteIfExists(File file) {
        try {
            Files.deleteIfExists(file.toPath());
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    public static void deleteIfExists(File file, ErrorHandler errorHandler) {
        try {
            Files.deleteIfExists(file.toPath());
        }
        catch (Exception ex) {
            errorHandler.onError(ex);
        }
    }

    public static FileChannel createEmptyFile(File file, long length) {
        return IoUtil.createEmptyFile(file, length, true);
    }

    public static FileChannel createEmptyFile(File file, long length, boolean fillWithZeros) {
        IoUtil.ensureDirectoryExists(file.getParentFile(), file.getParent());
        FileChannel templateFile = null;
        try {
            RandomAccessFile randomAccessFile = new RandomAccessFile(file, "rw");
            randomAccessFile.setLength(length);
            templateFile = randomAccessFile.getChannel();
            if (fillWithZeros) {
                IoUtil.fill(templateFile, 0L, length, (byte)0);
            }
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        return templateFile;
    }

    public static MappedByteBuffer mapExistingFile(File location, String descriptionLabel) {
        return IoUtil.mapExistingFile(location, FileChannel.MapMode.READ_WRITE, descriptionLabel);
    }

    public static MappedByteBuffer mapExistingFile(File location, String descriptionLabel, long offset, long length) {
        return IoUtil.mapExistingFile(location, FileChannel.MapMode.READ_WRITE, descriptionLabel, offset, length);
    }

    public static MappedByteBuffer mapExistingFile(File location, FileChannel.MapMode mapMode, String descriptionLabel) {
        IoUtil.checkFileExists(location, descriptionLabel);
        MappedByteBuffer mappedByteBuffer = null;
        try (RandomAccessFile file = new RandomAccessFile(location, IoUtil.getFileMode(mapMode));
             FileChannel channel = file.getChannel();){
            mappedByteBuffer = channel.map(mapMode, 0L, channel.size());
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        return mappedByteBuffer;
    }

    public static MappedByteBuffer mapExistingFile(File location, FileChannel.MapMode mapMode, String descriptionLabel, long offset, long length) {
        IoUtil.checkFileExists(location, descriptionLabel);
        MappedByteBuffer mappedByteBuffer = null;
        try (RandomAccessFile file = new RandomAccessFile(location, IoUtil.getFileMode(mapMode));
             FileChannel channel = file.getChannel();){
            mappedByteBuffer = channel.map(mapMode, offset, length);
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        return mappedByteBuffer;
    }

    public static MappedByteBuffer mapNewFile(File location, long length) {
        return IoUtil.mapNewFile(location, length, true);
    }

    public static MappedByteBuffer mapNewFile(File location, long length, boolean fillWithZeros) {
        Buffer mappedByteBuffer = null;
        try (FileChannel channel = FileChannel.open(location.toPath(), StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE);){
            mappedByteBuffer = channel.map(FileChannel.MapMode.READ_WRITE, 0L, length);
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        if (fillWithZeros) {
            int capacity = mappedByteBuffer.capacity();
            for (int pos = 0; pos < capacity; pos += 4096) {
                ((ByteBuffer)mappedByteBuffer).put(pos, (byte)0);
            }
        }
        return mappedByteBuffer;
    }

    public static void checkFileExists(File file, String name) {
        if (!file.exists()) {
            String msg = "missing file for " + name + " : " + file.getAbsolutePath();
            throw new IllegalStateException(msg);
        }
    }

    public static void unmap(MappedByteBuffer buffer) {
        BufferUtil.free(buffer);
    }

    public static long map(FileChannel fileChannel, FileChannel.MapMode mode, long offset, long length) {
        try {
            if (null != MappingMethods.MAP_FILE_DISPATCHER) {
                FileDescriptor fd = MappingMethods.GET_FILE_DESCRIPTOR.invoke(fileChannel);
                return MappingMethods.MAP_FILE_DISPATCHER.invoke(MappingMethods.FILE_DISPATCHER, fd, IoUtil.getMode(mode), offset, length, false);
            }
            if (null != MappingMethods.MAP_ADDRESS) {
                return MappingMethods.MAP_ADDRESS.invoke(fileChannel, IoUtil.getMode(mode), offset, length);
            }
            return MappingMethods.MAP_WITH_SYNC_ADDRESS.invoke(fileChannel, IoUtil.getMode(mode), offset, length, false);
        }
        catch (Throwable ex) {
            LangUtil.rethrowUnchecked(ex);
            return 0L;
        }
    }

    public static void unmap(FileChannel fileChannel, long address, long length) {
        try {
            if (null != MappingMethods.UNMAP_FILE_DISPATCHER) {
                MappingMethods.UNMAP_FILE_DISPATCHER.invoke(MappingMethods.FILE_DISPATCHER, address, length);
            } else {
                MappingMethods.UNMAP_ADDRESS.invoke(address, length);
            }
        }
        catch (Throwable ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    public static void unmap(ByteBuffer buffer) {
        if (buffer instanceof MappedByteBuffer) {
            IoUtil.unmap((MappedByteBuffer)buffer);
        }
    }

    public static String tmpDirName() {
        String tmpDirName = System.getProperty("java.io.tmpdir");
        if (!tmpDirName.endsWith(File.separator)) {
            tmpDirName = tmpDirName + File.separator;
        }
        return tmpDirName;
    }

    public static void removeTrailingSlashes(StringBuilder builder) {
        int lastCharIndex;
        char c;
        while (builder.length() > 1 && ('/' == (c = builder.charAt(lastCharIndex = builder.length() - 1)) || '\\' == c)) {
            builder.setLength(lastCharIndex);
        }
    }

    private static String getFileMode(FileChannel.MapMode mode) {
        return mode == FileChannel.MapMode.READ_ONLY ? "r" : "rw";
    }

    private static int getMode(FileChannel.MapMode mode) {
        if (mode == FileChannel.MapMode.READ_ONLY) {
            return 0;
        }
        if (mode == FileChannel.MapMode.READ_WRITE) {
            return 1;
        }
        return 2;
    }

    static class MappingMethods {
        static final MethodHandle MAP_FILE_DISPATCHER;
        static final MethodHandle UNMAP_FILE_DISPATCHER;
        static final Object FILE_DISPATCHER;
        static final MethodHandle GET_FILE_DESCRIPTOR;
        static final MethodHandle MAP_WITH_SYNC_ADDRESS;
        static final MethodHandle MAP_ADDRESS;
        static final MethodHandle UNMAP_ADDRESS;

        MappingMethods() {
        }

        private static Method getMethod(Class<?> klass, String name, Class<?> ... parameterTypes) throws NoSuchMethodException {
            Method method = klass.getDeclaredMethod(name, parameterTypes);
            method.setAccessible(true);
            return method;
        }

        static {
            try {
                Class<?> fileChannelClass = Class.forName("sun.nio.ch.FileChannelImpl");
                Class<?> fileDispatcherClass = Class.forName("sun.nio.ch.FileDispatcher");
                MethodHandles.Lookup lookup = MethodHandles.lookup();
                Object fileDispatcher = null;
                MethodHandle mapFileDispatcher = null;
                MethodHandle getFD = null;
                MethodHandle mapAddress = null;
                MethodHandle mapWithSyncAddress = null;
                MethodHandle unmapFileDispatcher = null;
                MethodHandle unmapAddress = null;
                try {
                    Field implLookupField = MethodHandles.Lookup.class.getDeclaredField("IMPL_LOOKUP");
                    lookup = (MethodHandles.Lookup)UnsafeAccess.UNSAFE.getObject(MethodHandles.Lookup.class, UnsafeAccess.UNSAFE.staticFieldOffset(implLookupField));
                    fileDispatcher = lookup.unreflectGetter(fileChannelClass.getDeclaredField("nd")).invoke();
                    getFD = lookup.unreflectGetter(fileChannelClass.getDeclaredField("fd"));
                    mapFileDispatcher = lookup.unreflect(fileDispatcherClass.getDeclaredMethod("map", FileDescriptor.class, Integer.TYPE, Long.TYPE, Long.TYPE, Boolean.TYPE));
                    unmapFileDispatcher = lookup.unreflect(fileDispatcherClass.getDeclaredMethod("unmap", Long.TYPE, Long.TYPE));
                }
                catch (Throwable ex) {
                    unmapAddress = lookup.unreflect(MappingMethods.getMethod(fileChannelClass, "unmap0", Long.TYPE, Long.TYPE));
                    try {
                        mapWithSyncAddress = lookup.unreflect(MappingMethods.getMethod(fileChannelClass, "map0", Integer.TYPE, Long.TYPE, Long.TYPE, Boolean.TYPE));
                    }
                    catch (Exception ex2) {
                        mapAddress = lookup.unreflect(MappingMethods.getMethod(fileChannelClass, "map0", Integer.TYPE, Long.TYPE, Long.TYPE));
                    }
                }
                MAP_FILE_DISPATCHER = mapFileDispatcher;
                UNMAP_FILE_DISPATCHER = unmapFileDispatcher;
                FILE_DISPATCHER = fileDispatcher;
                GET_FILE_DESCRIPTOR = getFD;
                MAP_WITH_SYNC_ADDRESS = mapWithSyncAddress;
                MAP_ADDRESS = mapAddress;
                UNMAP_ADDRESS = unmapAddress;
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }
    }
}

