package cn.sliew.milky.common.exception;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.*;

import static cn.sliew.milky.common.check.Ensures.checkNotNull;

/**
 * Utilities for dealing with throwables, stacks, etc.
 */
public final class ThrowableTraceFormater {
    /**
     * Stack filtering prefixes.
     */
    private final List<String> filteredPrefixes;

    /**
     * Default stack traces, no filtering.
     */
    public ThrowableTraceFormater() {
        this(Collections.<String>emptyList());
    }

    public ThrowableTraceFormater(List<String> filteredPrefixes) {
        this.filteredPrefixes = filteredPrefixes;
    }

    /**
     * Read the stacktrace of the supplied {@link Throwable} into a String.
     *
     * @param throwable the {@link Throwable} to be examined
     * @return the stack trace as generated by {@link Throwable#printStackTrace(java.io.PrintWriter)} method.
     */
    public static String readStackTrace(Throwable throwable) {
        checkNotNull(throwable, () -> "Throwable must not be null");

        StringWriter stringWriter = new StringWriter();
        try (PrintWriter printWriter = new PrintWriter(stringWriter)) {
            throwable.printStackTrace(printWriter);
        }
        return stringWriter.toString();
    }

    /**
     * Format an exception and all of its nested stacks into a string.
     */
    public String formatThrowable(Throwable t) {
        return formatThrowable(new StringBuilder(), t).toString();
    }

    /**
     * Format an exception and all of its nested stacks into a string.
     */
    public StringBuilder formatThrowable(StringBuilder b, Throwable t) {
        b.append(t.toString()).append("\n");
        formatStackTrace(b, t.getStackTrace());
        if (t.getCause() != null) {
            b.append("Caused by: ");
            formatThrowable(b, t.getCause());
        }
        return b;
    }

    /**
     * Format a list of stack entries into a string.
     */
    public StringBuilder formatStackTrace(StringBuilder b, StackTraceElement[] stackTrace) {
        return formatStackTrace(b, Arrays.asList(stackTrace));
    }

    public String formatStackTrace(StackTraceElement[] stackTrace) {
        return formatStackTrace(Arrays.asList(stackTrace));
    }

    public String formatStackTrace(Iterable<StackTraceElement> stackTrace) {
        return formatStackTrace(new StringBuilder(), stackTrace).toString();
    }

    /**
     * Format a list of stack entries into a string.
     */
    public StringBuilder formatStackTrace(StringBuilder b, Iterable<StackTraceElement> stackTrace) {
        Set<String> filteredSet = new HashSet<String>();
        for (StackTraceElement e : stackTrace) {
            String stackLine = e.toString();

            String filtered = null;
            for (String prefix : filteredPrefixes) {
                if (stackLine.startsWith(prefix)) {
                    filtered = prefix;
                    break;
                }
            }

            if (filtered != null) {
                if (filteredSet.isEmpty()) {
                    b.append("    [...");
                }
                filteredSet.add(filtered);
            } else {
                appendFiltered(b, filteredSet);
                b.append("    ").append(stackLine).append("\n");
            }
        }
        appendFiltered(b, filteredSet);
        return b;
    }

    private void appendFiltered(StringBuilder b, Set<String> filteredSet) {
        if (!filteredSet.isEmpty()) {
            boolean first = true;
            for (String prefix : filteredSet) {
                if (!first) {
                    b.append(", ");
                }
                first = false;
                b.append(prefix).append("*");
            }
            b.append("]\n");
            filteredSet.clear();
        }
    }
}
