package cn.sunxiansheng.log4j2.aspectj;

import cn.sunxiansheng.log4j2.annotation.SkipLogAspect;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;

/**
 * 日志切面
 *
 * @author sunxiansheng
 */
@Aspect
@Slf4j
public class LogAspect {

    private static final Gson GSON = new GsonBuilder()
            .setPrettyPrinting()
            .disableHtmlEscaping()
            .create();

    /**
     * ANSI 颜色代码
     */
    private static final String ANSI_RESET = "\u001B[0m";
    private static final String ANSI_RED = "\u001B[31m";
    private static final String ANSI_GREEN = "\u001B[32m";
    private static final String ANSI_YELLOW = "\u001B[33m";
    private static final String ANSI_BLUE = "\u001B[34m";
    private static final String ANSI_PURPLE = "\u001B[35m";
    private static final String ANSI_CYAN = "\u001B[36m";

    @Pointcut("execution(public * *..controller..*(..)) || " +
            "execution(public * *..service..*(..))")
    public void applicationPackagePointcut() {
        // 切点定义
    }

    @Around("applicationPackagePointcut()")
    public Object logAround(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();
        Class<?> targetClass = joinPoint.getTarget().getClass();

        // 检查是否存在 @SkipLogAspect 注解
        if (method.isAnnotationPresent(SkipLogAspect.class) ||
                targetClass.isAnnotationPresent(SkipLogAspect.class)) {
            // 直接执行目标方法，不记录日志
            return joinPoint.proceed();
        }

        String className = signature.getDeclaringTypeName();
        String methodName = signature.getName();
        Object[] args = joinPoint.getArgs();
        String requestParams = prettyJson(GSON.toJson(args)); // JSON 格式化

        long startTime = System.currentTimeMillis();

        // ✅ 进入方法日志
        StringBuilder enterLog = new StringBuilder();
        enterLog.append("\n")
                .append(ANSI_GREEN).append("==> 进入方法 ").append(ANSI_RESET).append(ANSI_PURPLE).append(methodName).append(ANSI_RESET).append("\n")
                .append(ANSI_BLUE).append("类名：").append(ANSI_CYAN).append(className).append(ANSI_RESET).append("\n")
                .append(ANSI_BLUE).append("参数：\n").append(ANSI_GREEN).append(requestParams).append(ANSI_RESET);
        log.info(enterLog.toString());

        Object result = joinPoint.proceed();

        long endTime = System.currentTimeMillis();
        String response = prettyJson(GSON.toJson(result)); // JSON 格式化

        // ✅ 退出方法日志
        StringBuilder exitLog = new StringBuilder();
        exitLog.append("\n")
                .append(ANSI_RED).append("<== 退出方法 ").append(ANSI_RESET).append(ANSI_PURPLE).append(methodName).append(ANSI_RESET).append("\n")
                .append(ANSI_BLUE).append("类名：").append(ANSI_CYAN).append(className).append(ANSI_RESET).append("\n")
                .append(ANSI_BLUE).append("耗时：").append(ANSI_GREEN).append(endTime - startTime).append(" ms").append(ANSI_RESET).append("\n")
                .append(ANSI_BLUE).append("返回值：\n").append(ANSI_GREEN).append(response).append(ANSI_RESET);
        log.info(exitLog.toString());

        return result;
    }

    /**
     * 美化 JSON 输出，使其格式化但不带 `|`
     */
    private String prettyJson(String json) {
        try {
            JsonElement jsonElement = JsonParser.parseString(json);
            return new GsonBuilder().setPrettyPrinting().create().toJson(jsonElement);
        } catch (Exception e) {
            return json; // 如果 JSON 解析失败，直接返回原字符串
        }
    }
}