package cn.sunxiansheng.tool.response;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.ToString;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Description: 响应状态码和信息的枚举类，用于统一管理响应状态码及其对应的消息。
 * 通过枚举类的常量可以快速获取响应码及消息。
 * 还提供了通过响应码查找对应枚举的功能。
 *
 * @Author sun
 * @Create 2024/10/11 18:23
 * @Version 1.0
 */
@Getter
@AllArgsConstructor
@ToString
public enum RespBeanEnum {

    // ============================== 枚举常量 ==============================
    // 通用响应状态
    SUCCESS(200, "操作成功"), // 成功响应
    ERROR(500, "系统异常"),   // 错误响应
    // 可以在此处添加更多枚举常量，例如：
    // NOT_FOUND(404, "资源未找到"),  // 资源未找到
    // UNAUTHORIZED(401, "未授权"),  // 未授权访问
    ;
    // ============================== 枚举常量 ==============================

    // 响应码和响应信息
    private final Integer code;   // 响应状态码
    private final String message; // 响应信息

    /**
     * 将枚举转换为 Map（静态初始化），用于根据响应码快速查找对应的枚举值。
     * <p>
     * 该 Map 存储了响应码和对应枚举的映射关系，以便高效查找。
     * </p>
     */
    public static final Map<Integer, RespBeanEnum> channelEnumMap = Stream.of(RespBeanEnum.values())
            .collect(Collectors.toMap(RespBeanEnum::getCode, Function.identity()));

    /**
     * 根据响应码获取对应的枚举。
     *
     * @param code 响应码
     * @return 对应的枚举，如果没有找到则返回 null
     */
    public static RespBeanEnum getByCode(int code) {
        return channelEnumMap.get(code);
    }
}
