package cn.sunxiansheng.tool.response;

import lombok.Data;

import java.io.Serializable;

/**
 * Description: 通用响应封装类，通过枚举来获取响应的 code 和 message。
 *              提供了构建响应对象的建造者模式，支持链式调用。
 *
 * @Author sun
 * @Create 2024/10/11
 * @Version 1.0
 */
@Data
public class ResultWrapper<T> implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * 是否成功
     */
    private boolean success;

    /**
     * 响应代码
     */
    private int code;

    /**
     * 响应消息
     */
    private String message;

    /**
     * 响应数据
     */
    private T data;

    /**
     * 私有构造器，防止外部直接创建对象。
     */
    private ResultWrapper() {
    }

    /**
     * 内部建造者类，负责构建 {@link ResultWrapper} 对象。
     *
     * @param <T> 响应数据类型
     */
    public static class Builder<T> {

        private boolean success;
        private int code;
        private String message;
        private T data;

        /**
         * 设置是否成功
         *
         * @param success 是否成功
         * @return 当前建造者对象，用于链式调用
         */
        public Builder<T> success(boolean success) {
            this.success = success;
            return this;
        }

        /**
         * 设置响应代码
         *
         * @param code 响应代码
         * @return 当前建造者对象，用于链式调用
         */
        public Builder<T> code(int code) {
            this.code = code;
            return this;
        }

        /**
         * 设置响应消息
         *
         * @param message 响应消息
         * @return 当前建造者对象，用于链式调用
         */
        public Builder<T> message(String message) {
            this.message = message;
            return this;
        }

        /**
         * 设置响应数据
         *
         * @param data 响应数据
         * @return 当前建造者对象，用于链式调用
         */
        public Builder<T> data(T data) {
            this.data = data;
            return this;
        }

        /**
         * 构建并返回一个 {@link ResultWrapper} 实例。
         *
         * @return 构建的 {@link ResultWrapper} 对象
         */
        public ResultWrapper<T> build() {
            ResultWrapper<T> result = new ResultWrapper<>();
            result.success = this.success;
            result.code = this.code;
            result.message = this.message;
            result.data = this.data;
            return result;
        }
    }

    // ============================== 快捷方法 ==============================

    /**
     * 返回一个表示成功的响应对象。
     *
     * @param <T> 响应数据类型
     * @return 成功的 {@link ResultWrapper} 对象
     */
    public static <T> ResultWrapper<T> ok() {
        return new Builder<T>()
                .success(true)
                .code(RespBeanEnum.SUCCESS.getCode())
                .message(RespBeanEnum.SUCCESS.getMessage())
                .build();
    }

    /**
     * 返回一个表示成功的响应对象，并携带数据。
     *
     * @param <T> 响应数据类型
     * @param data 响应数据
     * @return 成功的 {@link ResultWrapper} 对象，包含数据
     */
    public static <T> ResultWrapper<T> ok(T data) {
        return new Builder<T>()
                .success(true)
                .code(RespBeanEnum.SUCCESS.getCode())
                .message(RespBeanEnum.SUCCESS.getMessage())
                .data(data)
                .build();
    }

    /**
     * 返回一个表示成功的响应对象，并携带自定义消息。
     *
     * @param <T> 响应数据类型
     * @param data 响应数据
     * @param message 自定义消息
     * @return 成功的 {@link ResultWrapper} 对象，包含数据和消息
     */
    public static <T> ResultWrapper<T> ok(T data, String message) {
        return new Builder<T>()
                .success(true)
                .code(RespBeanEnum.SUCCESS.getCode())
                .message(message)
                .data(data)
                .build();
    }

    /**
     * 返回一个表示失败的响应对象。
     *
     * @param <T> 响应数据类型
     * @return 失败的 {@link ResultWrapper} 对象
     */
    public static <T> ResultWrapper<T> fail() {
        return new Builder<T>()
                .success(false)
                .code(RespBeanEnum.ERROR.getCode())
                .message(RespBeanEnum.ERROR.getMessage())
                .build();
    }

    /**
     * 返回一个表示失败的响应对象，并携带自定义消息。
     *
     * @param <T> 响应数据类型
     * @param message 自定义消息
     * @return 失败的 {@link ResultWrapper} 对象，包含自定义消息
     */
    public static <T> ResultWrapper<T> fail(String message) {
        return new Builder<T>()
                .success(false)
                .code(RespBeanEnum.ERROR.getCode())
                .message(message)
                .build();
    }

    /**
     * 返回一个表示失败的响应对象，并携带自定义代码和消息。
     *
     * @param <T> 响应数据类型
     * @param code 自定义错误代码
     * @param message 自定义错误消息
     * @return 失败的 {@link ResultWrapper} 对象，包含自定义代码和消息
     */
    public static <T> ResultWrapper<T> fail(int code, String message) {
        return new Builder<T>()
                .success(false)
                .code(code)
                .message(message)
                .build();
    }

    /**
     * 返回一个表示失败的响应对象，并携带自定义代码、消息和数据。
     *
     * @param <T> 响应数据类型
     * @param code 自定义错误代码
     * @param message 自定义错误消息
     * @param data 响应数据
     * @return 失败的 {@link ResultWrapper} 对象，包含自定义代码、消息和数据
     */
    public static <T> ResultWrapper<T> fail(int code, String message, T data) {
        return new Builder<T>()
                .success(false)
                .code(code)
                .message(message)
                .data(data)
                .build();
    }

    /**
     * 返回一个表示失败的响应对象，并携带 {@link RespBeanEnum} 定义的错误代码和消息。
     *
     * @param <T> 响应数据类型
     * @param respBeanEnum 错误枚举类型
     * @return 失败的 {@link ResultWrapper} 对象，包含枚举定义的错误代码和消息
     */
    public static <T> ResultWrapper<T> fail(RespBeanEnum respBeanEnum) {
        return new Builder<T>()
                .success(false)
                .code(respBeanEnum.getCode())
                .message(respBeanEnum.getMessage())
                .build();
    }

    // ============================== 快捷方法 ==============================
}
