package cn.sunxiansheng.tool.response;

import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 处理与 Servlet 相关的响应工具类，主要用于在过滤器中响应数据，避免被全局异常处理器捕获。
 * 提供了封装好的方法来返回成功和失败的 JSON 响应。
 *
 * @Author sun
 * @Create 2024/10/11
 * @Version 1.0
 */
@Slf4j  // 启用日志功能
public class SunRaysServletResponse {

    // 使用 Gson 来将对象转化为 JSON 字符串
    private static final Gson gson = new Gson();

    /**
     * 将构造好的响应数据写入 HttpServletResponse 输出流，返回 JSON 格式的响应。
     *
     * @param response HttpServletResponse 对象
     * @param success 响应是否成功
     * @param code 响应状态码
     * @param message 响应消息
     * @param data 响应数据
     * @param <T> 响应数据类型
     * @throws IOException 如果写入响应时发生 I/O 异常
     */
    public static <T> void writeResponse(HttpServletResponse response, boolean success, int code, String message, T data) throws IOException {
        // 设置响应内容类型为 JSON
        response.setContentType("application/json");
        response.setCharacterEncoding("UTF-8");

        // 构造 ResultWrapper 对象
        ResultWrapper<T> resultWrapper = new ResultWrapper.Builder<T>()
                .success(success)
                .code(code)
                .message(message)
                .data(data)
                .build();

        // 将 ResultWrapper 对象转化为 JSON 字符串
        String jsonResponse = gson.toJson(resultWrapper);

        // 打印日志
        log.info("SunRaysServletResponse Data: {}", jsonResponse);

        // 将 JSON 响应写入输出流
        response.getWriter().write(jsonResponse);
    }

    /**
     * 返回一个表示失败的 JSON 响应，写入到 HttpServletResponse。
     *
     * @param response HttpServletResponse 对象
     * @param code 错误状态码
     * @param message 错误信息
     * @throws IOException 如果写入响应时发生 I/O 异常
     */
    public static void writeFail(HttpServletResponse response, int code, String message) throws IOException {
        // 打印日志
        log.error("Fail SunRaysServletResponse: code={}, message={}", code, message);

        // 调用通用的 writeResponse 方法，构造失败响应
        writeResponse(response, false, code, message, null);
    }

    /**
     * 返回一个表示成功的 JSON 响应，写入到 HttpServletResponse。
     *
     * @param response HttpServletResponse 对象
     * @param code 响应状态码
     * @param message 响应消息
     * @param data 响应数据
     * @param <T> 响应数据类型
     * @throws IOException 如果写入响应时发生 I/O 异常
     */
    public static <T> void writeSuccess(HttpServletResponse response, int code, String message, T data) throws IOException {
        // 打印日志
        log.info("Success SunRaysServletResponse: code={}, message={}, data={}", code, message, data);

        // 调用通用的 writeResponse 方法，构造成功响应
        writeResponse(response, true, code, message, data);
    }
}
