package cn.sunxiansheng.tool.utils;

import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Description: 请求相关的工具类，提供了从请求头中获取信息、读取请求体内容等功能。
 * 该类包含用于操作 HTTP 请求的一些常用方法。
 *
 * @Author sun
 * @Create 2025/1/25 12:31
 * @Version 1.0
 */
@Slf4j
public class RequestUtil {

    /**
     * 从请求头中获取指定的 header 值
     *
     * @param request HttpServletRequest 请求对象
     * @param headerName 要获取的请求头名称
     * @return 请求头的值，如果不存在则返回 null
     * @throws IllegalArgumentException 如果请求对象或头名称为空，则返回 null
     */
    public static String getHeader(HttpServletRequest request, String headerName) {
        if (request == null || headerName == null || headerName.isEmpty()) {
            return null;
        }
        return request.getHeader(headerName);
    }

    /**
     * 获取指定的多个请求头
     *
     * @param request HttpServletRequest 请求对象
     * @param headerNames 要获取的请求头名称数组
     * @return 请求头的值映射，若某个请求头不存在则对应值为 null
     */
    public static Map<String, String> getHeaders(HttpServletRequest request, String... headerNames) {
        Map<String, String> headers = new HashMap<>();
        for (String headerName : headerNames) {
            String headerValue = request.getHeader(headerName);
            headers.put(headerName, headerValue);
        }
        return headers;
    }

    /**
     * 获取原始的 Body 数据
     * <p>
     * 从 HTTP 请求中读取原始的 Body 内容，并返回为字符串。常用于读取 JSON 或其他请求体数据。
     * </p>
     *
     * @param request HttpServletRequest 请求对象
     * @return 请求的原始 Body 数据，可能为空字符串
     */
    public static String getBody(HttpServletRequest request) {
        // 获取原始报文主体
        StringBuilder bodyBuilder = new StringBuilder();
        try (BufferedReader reader = request.getReader()) {
            String line;
            while ((line = reader.readLine()) != null) {
                bodyBuilder.append(line).append("\n");
            }
        } catch (IOException e) {
            log.error("获取原始的 Body 数据失败！", e);
        }

        String body = bodyBuilder.toString().trim();

        // 打印原始报文（假设 JsonUtil 是一个 JSON 格式化工具）
        log.info("原始报文: {}", JsonUtil.toPrettyJson(body));
        return body;
    }
}
