package cn.takujo.common_api.log;

import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import lombok.extern.slf4j.Slf4j;

/**
 * 日志切面，继承后加spring的切面和组件注解 注：如用非shiro安全框架
 * ，则还需重写AuthenticatedAndPermitted,getAdministrator方法
 * 
 * @author wzx
 *
 */
@Slf4j
public abstract class LogAspect {

	@Pointcut("@annotation(cn.takujo.common_api.log.Logable)")
	public void webPointCut() {
	}

	@AfterThrowing("webPointCut()")
	public void throwss(JoinPoint jp) {
		log.warn("LogAspect afterThrowing: exception erro");
	}

	@Around("webPointCut()")
	public Object arround(ProceedingJoinPoint pjp) {
		try {
			Object o = pjp.proceed();
			LogData logData = handleService(pjp, o);
			logPersistence(logData);
			log.info("save logdata success");
			return o;
		} catch (Throwable e) {
			log.warn("LogAspect arround: exception erro");
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * 访问当前路径所需要的权限
	 * 
	 * @param uri
	 *            请求路径
	 * @return 权限描述
	 */
	protected abstract String uriToPermStr(String uri);

	/**
	 * 日志持久化
	 * 
	 * @param log
	 *            日志信息
	 */
	protected abstract void logPersistence(LogData log);

	/**
	 * 认证和授权到日志 补充：OperaterId，OperaterName，Allow到日志
	 * 默认使用shiro框架实现，如用其他安全框架，则需重写该方法，并补全以上信息到日志中
	 * 
	 * @param logData
	 *            日志信息
	 * @param administrator
	 *            管理员实体
	 * @param permStr
	 *            所需权限
	 */
	protected void AuthenticatedAndPermitted(LogData logData, Administrator administrator, String permStr) {
		Integer id = administrator.getId();
		String name = administrator.getName();
		Subject subject = SecurityUtils.getSubject();
		boolean authenticated = subject.isAuthenticated();
		if (authenticated) {
			logData.setOperaterId(id);
			logData.setOperaterName(name);
			if (subject.isPermitted(permStr)) {
				logData.setAllow(1);
			} else {
				logData.setAllow(0);
			}
		} else {
			logData.setAllow(0);
			logData.setOperaterId(0);
			logData.setOperaterName("未知人员");
		}
	}

	/**
	 * 返回管理员实体 默认使用shiro框架实现，如用其他安全框架，则需重写该方法
	 * 
	 * @return 实体
	 */
	protected Administrator getAdministrator() {
		Subject subject = SecurityUtils.getSubject();
		Principal pri = (Principal) subject.getPrincipal();
		return pri.getAdministrator();
	}

	private Map<String, Object> handleRequest() {
		ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
		HttpServletRequest request = attributes.getRequest();
		String uri = request.getRequestURI();
		String visitIp = request.getRemoteAddr();
		String permStr = uriToPermStr(uri);
		Map<String, Object> map = new HashMap<>();
		map.put("visitIp", visitIp);
		map.put("permStr", permStr);
		return map;
	}

	private LogData handleService(ProceedingJoinPoint pjp, Object result) {
		LogData logData = new LogData();
		logData.setTime(new Date());
		Map<String, Object> handleRequest = handleRequest();
		Object vipobject = handleRequest.get("visitIp");
		Object perobject = handleRequest.get("permStr");
		String permStr = null;
		if (vipobject != null) {
			logData.setVisitIp((String) vipobject);
		}
		if (perobject != null) {
			permStr = (String) perobject;
		}
		AuthenticatedAndPermitted(logData, getAdministrator(), permStr);
		Signature signature = pjp.getSignature();
		Class<?> declaringType = signature.getDeclaringType();
		String methodName = signature.getName();
		Method method = getMethod(declaringType, methodName);
		if (method != null) {
			try {
				Logable annotation = method.getAnnotation(Logable.class);
				if (annotation != null) {
					logData.setAction(annotation.action());
					logData.setType(annotation.type());
				}
				StringBuffer buffer = new StringBuffer("");
				Object[] args = pjp.getArgs();
				for (Object obj : args) {
					if (obj instanceof LogRecord) {
						LogRecord lr = (LogRecord) obj;
						buffer.append(lr.toLog() + ",");
					} else if (obj instanceof Integer) {
						buffer.append(obj.toString() + ",");
					}
				}
				int lastIndexOf = buffer.lastIndexOf(",");
				if (lastIndexOf != -1) {
					buffer.deleteCharAt(lastIndexOf);
				}
				logData.setThing(buffer.toString());
				if (result == null) {
					logData.setResult(0);
				} else {
					logData.setResult(1);
					if (result instanceof Integer) {
						Integer i = (Integer) result;
						if (i == 0) {
							logData.setResult(0);
						}
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
				logData.setResult(0);
			}
		}
		return logData;
	}

	private Method getMethod(Class<?> type, String methodName) {
		Method[] declaredMethods = type.getDeclaredMethods();
		for (Method method : declaredMethods) {
			if (method.getName().equals(methodName)) {
				return method;
			}
		}
		return null;
	}

}
