package cn.takujo.common_api.util;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.imageio.ImageIO;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.Binarizer;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.EncodeHintType;
import com.google.zxing.LuminanceSource;
import com.google.zxing.MultiFormatReader;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;

/**
 * 二维码工具
 * 
 * @author wzx
 *
 */
public final class QrcoderUtil {

	public static final int BLACK = 0x000000;
	public static final int WHITE = 0xFFFFFF;
	public static final int RED = 0xFF0000;
	public static final int YELLOW = 0xFFFF00;
	public static final int BLUE = 0x0000FF;
	public static final int PURPLE = 0x800080;
	public static final int GREEN = 0x008000;

	/**
	 * 生成二维码（默认黑白像素点）
	 * 
	 * @param content
	 *            要生成的内容
	 * @param width
	 *            宽度
	 * @param height
	 *            高度
	 * @param outPutPath
	 *            输出全路径(包含文件本身带扩展名(图片格式)) 备注：文件夹不存在则会创建
	 * @throws Exception
	 *             二维码生成失败
	 */
	public static void generate(String content, int width, int height, String outPutPath) throws Exception {
		Map<EncodeHintType, String> his = new HashMap<EncodeHintType, String>();
		his.put(EncodeHintType.CHARACTER_SET, "utf-8");
		try {
			BitMatrix encode = new MultiFormatWriter().encode(content, BarcodeFormat.QR_CODE, width, height, his);
			int codeWidth = encode.getWidth();
			int codeHeight = encode.getHeight();
			BufferedImage image = new BufferedImage(codeWidth, codeHeight, BufferedImage.TYPE_INT_RGB);
			for (int i = 0; i < codeWidth; i++) {
				for (int j = 0; j < codeHeight; j++) {
					image.setRGB(i, j, encode.get(i, j) ? BLACK : WHITE);
				}
			}
			File outPutImage = new File(outPutPath);
			String fileName = outPutImage.getName();
			String extension = fileName.substring(fileName.lastIndexOf(".") + 1);
			outPutImage.getParentFile().mkdirs();
			if (!outPutImage.exists()) {
				outPutImage.createNewFile();
			}
			ImageIO.write(image, extension, outPutImage);
		} catch (WriterException e) {
			throw new Exception("二维码生成失败");
		} catch (IOException e) {
			throw new Exception("二维码生成失败");
		} catch (Exception e) {
			throw new Exception("二维码生成失败");
		}
	}

	/**
	 * 生成二维码（自定义像素颜色）
	 * 
	 * @param content
	 *            要生成的内容
	 * @param width
	 *            宽度
	 * @param height
	 *            高度
	 * @param outPutPath
	 *            输出全路径(包含文件本身带扩展名(图片格式)) 备注：文件夹不存在则会创建
	 * @param frontColor
	 *            前台颜色(十六进制) 例 0x0000FF
	 * @param backColor
	 *            后台颜色(十六进制) 例 0xFFC0CB
	 * @throws Exception
	 *             二维码生成失败
	 */
	public static void generate(String content, int width, int height, String outPutPath, int frontColor, int backColor)
			throws Exception {
		Map<EncodeHintType, String> his = new HashMap<EncodeHintType, String>();
		his.put(EncodeHintType.CHARACTER_SET, "utf-8");
		try {
			BitMatrix encode = new MultiFormatWriter().encode(content, BarcodeFormat.QR_CODE, width, height, his);
			int codeWidth = encode.getWidth();
			int codeHeight = encode.getHeight();
			BufferedImage image = new BufferedImage(codeWidth, codeHeight, BufferedImage.TYPE_INT_RGB);
			for (int i = 0; i < codeWidth; i++) {
				for (int j = 0; j < codeHeight; j++) {
					image.setRGB(i, j, encode.get(i, j) ? frontColor : backColor);
				}
			}
			File outPutImage = new File(outPutPath);
			String fileName = outPutImage.getName();
			String extension = fileName.substring(fileName.lastIndexOf(".") + 1);
			outPutImage.getParentFile().mkdirs();
			if (!outPutImage.exists()) {
				outPutImage.createNewFile();
			}
			ImageIO.write(image, extension, outPutImage);
		} catch (WriterException e) {
			throw new Exception("二维码生成失败");
		} catch (IOException e) {
			throw new Exception("二维码生成失败");
		} catch (Exception e) {
			throw new Exception("二维码生成失败");
		}
	}

	/**
	 * 解析二维码
	 * 
	 * @param codePath
	 *            二维码路径（完整）
	 * @return 解析内容
	 * @throws Exception
	 *             二维码不存在或解析失败
	 */
	public static String analyze(String codePath) throws Exception {
		File file = new File(codePath);
		if (!file.exists()) {
			throw new Exception("二维码不存在,无法解析");
		} else {
			try {
				BufferedImage image = ImageIO.read(file);
				LuminanceSource source = new BufferedImageLuminanceSource(image);
				Binarizer binarizer = new HybridBinarizer(source);
				BinaryBitmap binaryBitmap = new BinaryBitmap(binarizer);
				Map<DecodeHintType, String> hints = new HashMap<>();
				hints.put(DecodeHintType.CHARACTER_SET, "utf-8");
				MultiFormatReader formatReader = new MultiFormatReader();
				Result result = formatReader.decode(binaryBitmap, hints);
				return result.getText();
			} catch (NotFoundException e) {
				throw new Exception("解析失败");
			} catch (IOException e) {
				throw new Exception("解析失败");
			} catch (Exception e) {
				throw new Exception("解析失败");
			}
		}
	}

}
