/*
 * Copyright (c) 2017 Beijing Tiande Technology Co., Ltd.
 * All Rights Reserved.
 */
package cn.tdchain.jbcc;

import cn.tdchain.Trans;
import cn.tdchain.cipher.rsa.Sha256Util;
import cn.tdchain.cipher.utils.HashCheckUtil;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Trans util
 *
 * @author xiaoming
 * 2019年4月18日
 */
public class TransUtil {
    public final static int HASH_LENGTH = 64;

    /**
     * get a trans hash
     *
     * @param tx
     * @return String
     */
    public static String getTransHash(Trans tx) {
        JSONObject txJsonO = (JSONObject) JSONObject.toJSON(tx);
        txJsonO.remove("hash");//不包含hash字段
        txJsonO.remove("blockHash");//不包含hash字段
        txJsonO.remove("connectionId");//不包含connectionId字段
        txJsonO.remove("status");//不包含status字段
        txJsonO.remove("msg");//不包含msg字段
        txJsonO.remove("index");//不包含index字段
        txJsonO.remove("preHash");//不包含preHash字段
        txJsonO.remove("author");//不包含author字段
        txJsonO.remove("version");//不包含version字段
        txJsonO.remove("height");//不包含height字段
        txJsonO.remove("signMap"); //不包含signMap字段（暂时？）
        
        //兼容2.0.3以前版本hash计算
        if(null == tx.getSignMap() && null == tx.getCategory() && null == tx.getKeyVersionMap()) {
        	//2.0.3以前的交易，不对set sign_map, key_version_map, category字段hash
        	txJsonO.remove("signMap"); //不包含signMap字段
        	txJsonO.remove("keyVersionMap"); //不包含keyVersionMap字段
        	txJsonO.remove("category"); //不包含category字段
        }
        
        
        String txJsonStr = txJsonO.toJSONString();
        String hash = Sha256Util.hash(txJsonStr);
        return hash;
    }

    public static void verifyContratTrans(Trans trans) {
        if (trans == null) {
            throw new TransInfoException("trans is null");
        }
        if (StringUtils.isBlank(trans.getAccount())) {
            throw new TransInfoException("trans account should init");
        }
        if (StringUtils.isBlank(trans.getData())) {
            throw new TransInfoException("trans data is blank");
        }
        if (trans.getTimestamp() == null) {
            throw new TransInfoException("trans timestamp is null");
        }
    }

    /**
     * Description: 解析块高度：64位hash"+"height
     *
     * @param hashHeight
     * @return Long
     */
    public static Long getHeight(String hashHeight) {
        if (StringUtils.isBlank(hashHeight) || hashHeight.length() <= HASH_LENGTH)
            throw new RuntimeException("getHeight: split hash height exception, hash string: " + hashHeight);

        Long height = 0L;
        try {
            height = Long.valueOf(hashHeight.substring(HASH_LENGTH));
        } catch (Exception e) {
            throw new RuntimeException("getHeight: split hash height exception, hash string: " + hashHeight);
        }
        return height;
    }

    public static Long getHeight(Trans trans) {
        if (trans == null)
            throw new RuntimeException("trans is null");

        String hashHeight = trans.getHash();
        return getHeight(hashHeight);
    }

    /**
     * Description: 解析hash串：64位hash"+"height
     *
     * @param hashHeight
     * @return String
     */
    public static String getHash(String hashHeight) {
        if (StringUtils.isBlank(hashHeight) || hashHeight.length() <= HASH_LENGTH)
            throw new RuntimeException("split hash string exception, hash string: " + hashHeight);

        String hash = "";
        try {
            hash = hashHeight.substring(0, HASH_LENGTH);
        } catch (Exception e) {
            throw new RuntimeException("getHeight: split hash string exception, hash string: " + hashHeight);
        }

        return hash;
    }

    public static String getHash(Trans trans) {
        if (trans == null)
            throw new RuntimeException("trans is null");

        String hashHeight = trans.getHash();
        return getHash(hashHeight);
    }

    /**
     * hash 规则：64位hash "+" height
     *
     * @param hash
     * @param height
     * @return String
     */
    public static String hashHeight(String hash, Long height) {
        if (StringUtils.isBlank(hash) || !HashCheckUtil.hashCheck(hash))
            throw new TransInfoException("trans hash is error, hash=" + hash);

        if (height == null || height < 0)
            throw new TransInfoException("block height is null or less zero");

        return hash.concat(height + "");
    }

    public static void main(String[] args) {
        String str = "937d45babcf5fac3a3b889957cfd706c8ce1d1e5542acecfca8e1764a3e068b2";
        System.out.println(getHeight(str));
        System.out.println(getHash(str));
    }

    public static String getKeyAndAccountStr(String key, String account) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(account);
        return key + "&" + account;
    }

    public static Set<String> getBatchKeyAndAccountStrs(BatchTrans<? extends Trans> batchTrans) {
        if (batchTrans == null) {
            return new HashSet<>();
        }
        if (CollectionUtils.isEmpty(batchTrans.getTransSet())) {
            return new HashSet<>();
        }
        Set<String> result = batchTrans.getTransSet().stream().map(k -> getKeyAndAccountStr(k.getKey(), k.getAccount())).collect(Collectors.toSet());
        return result;
    }

    public static boolean verifyKey(String key) {
        // 交易属性验证
        if (StringUtils.isBlank(key) || key.length() > 64) {
            throw new TransInfoException("key is null or too long, max size is 64.");
        }

        // 非贪婪判断包含空白字符
        if (key.matches(".*?\\s+?.*?")) {
            throw new TransInfoException("key:{" + key + "} has space character.");
        }

        // key不能是纯数字
        if (key.matches("\\d+")) {
            throw new TransInfoException("key:{" + key + "} should not numbers only.");
        }

        //key不能有特殊字符
        if (HashCheckUtil.illegalCharacterCheck(key)) {
            throw new TransInfoException("key have Illegal character.");
        }
        return true;
    }

}
