/*
 * Copyright (c) 2017 Beijing Tiande Technology Co., Ltd.
 * All Rights Reserved.
 */
package cn.tdchain.jbcc;

import cn.tdchain.Block;
import cn.tdchain.Trans;
import cn.tdchain.TransHead;
import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.CipherException;
import cn.tdchain.cipher.Key;
import cn.tdchain.cipher.utils.HashCheckUtil;
import cn.tdchain.jbcc.net.NetType;
import cn.tdchain.jbcc.net.Net;
import cn.tdchain.jbcc.net.info.Node;
import cn.tdchain.jbcc.net.io.IONet;
import cn.tdchain.jbcc.net.nio.NioNet;
import cn.tdchain.jbcc.rpc.RPCMessage;
import cn.tdchain.jbcc.rpc.RPCResult;
import cn.tdchain.tdmsp.Msp;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static cn.tdchain.jbcc.ConnectionFactory.DEFAULT_TIMEOUT;
import static cn.tdchain.jbcc.ConnectionFactory.MAX_TRANS_COUNT;
import static cn.tdchain.jbcc.ConnectionFactory.MAX_TRANS_HISTORY_COUNT;
import static cn.tdchain.jbcc.TransUtil.HASH_LENGTH;

/**
 * function：description
 * datetime：2019-03-27 13:07
 * author：warne
 */
public class Connection {
    /**
     * Description: 添加一笔交易
     *
     * @param trans
     * @return TransHead
     */
    public TransHead addTrans(Trans trans) {
        //# 判断交易null
        if (trans == null)
            throw new BatchTransException("trans is null");

        BatchTrans<Trans> batch = new BatchTrans<>();
        batch.setConnectionId(this.connectionId);
        batch.addTransToBatch(trans);

        BatchTrans<TransHead> batchTrans = addBatchTrans(batch);
        return batchTrans.oneTrans();
    }

    /**
     * Description: 批量交易
     *
     * @param batch
     * @return BatchTrans<TransHead>
     */
    public BatchTrans<TransHead> addBatchTrans(BatchTrans<Trans> batch) {
        //# 判断批量交易null
        if (batch == null || (batch.getTransSet() == null || batch.getTransSet().size() == 0))
            throw new BatchTransException("batch is empty ");

        batch.setConnectionId(this.connectionId);
        batch.check();

        RPCMessage msg = getMessage();
        msg.setMessageId(batch.getId());
        msg.setMsg(batch.toJsonString());
        msg.setTargetType(RPCMessage.TargetType.TX_WAIT);

        // 异步将交易发送给请求搬运工
        net.request(msg);

        // 异步等待交易返回
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 12000);// 12秒超时，如果共识超时也能知道交易是否成功

        /** 处理返回结果 */
        int succesCount = 0;
        int failCount = 0;
        BatchTrans<TransHead> succesBatch = null;
        BatchTrans<TransHead> failBatch = new BatchTrans<TransHead>();
        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;

            BatchTrans<TransHead> tempBatch = null;
            if (result.getType() == RPCResult.ResultType.tx_status) {
                try {
                    // 反序列化可能异常
                    tempBatch = JSON.parseObject(result.getEntity(), new TypeReference<BatchTrans<TransHead>>() {
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (tempBatch != null) {
                    succesCount++;// 成功一次
                    succesBatch = tempBatch;// 记住成功的交易对象
                } else {
                    failCount++;// 失败一次
                }
            }
        }

        if (succesCount >= this.minSucces || succesCount > failCount) {
            return succesBatch;// 返回成功的交易对象
        } else {
            failBatch.setStatus(Trans.TransStatus.failed);
            return failBatch;
        }
    }

    /**
     * Description: 根据块高度查询块信息
     *
     * @param height
     * @return Block<Trans>
     */
    public Block<Trans> getBlock(Long height) {
        Block<Trans> b = null;
        if (height == null || (height < 1 && height != -1))
            return b;

        Map<String, String> command = new HashMap<>();
        command.put("height", height.toString());
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_BLOCK);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 10000);
        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;

            if (result.getType() == RPCResult.ResultType.block) {
                try {
                    b = JSON.parseObject(result.getEntity(), new TypeReference<Block<Trans>>() {
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (b != null) {
                    try {
                        b.check();
                    } catch (BlockException e) {
                        e.printStackTrace();
                    }
                }
            }
        }

        return b;
    }


    /**
     * Description: 根据height查询block header
     *
     * @param height
     * @return Block
     */
    public Block getBlockHeaderByHeight(Long height) {
        Block b = null;
        Map<String, String> command = new HashMap<String, String>();
        command.put("height", height.toString());

        RPCMessage msg = getMessage();
        msg.setCommand(command);
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_BLOCK_HEADER);

        net.request(msg);

        //resphonse
        List<RPCResult> r_list = net.resphone(msg.getMessageId(), 10000);

        for (RPCResult r : r_list) {
            if (r != null && (r.getType() == RPCResult.ResultType.block_header) && (r.getEntity() != null && !"".equals(r.getEntity()))) {
                try {
                    b = JSON.parseObject(r.getEntity(), new TypeReference<Block<Trans>>() {
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }
                if (b != null) {
                    return b;
                }

            }
        }
        return null;

    }


    /**
     * Description: 查询最大块信息
     *
     * @return Block<Trans>
     */
    public Block<Trans> getMaxBlock() {
        return getBlock(-1L);
    }

    /**
     * Description: 根据hash查询交易信息
     *
     * @param hash
     * @return Trans
     */
    public Trans getTransByHash(String hash) {
        if (!HashCheckUtil.hashCheck(hash))
            return null;

        if (hash.length() <= HASH_LENGTH)
            return null;

        Map<String, String> command = new HashMap<>();
        command.put("hash", hash);
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_HASH);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 3000);

        return transByResult(rpcResultList);
    }

    /**
     * Description: 根据hashlist查询交易
     *
     * @param hashList
     * @return List<Trans>
     */
    public List<Trans> getTransListByHashList(List<String> hashList) {
        List<Trans> transList = null;
        if (hashList == null || hashList.size() == 0)
            return new ArrayList<>();

        if (hashList.size() > MAX_TRANS_COUNT)
            throw new BatchTransException("hashList is too large [ less than or equal to " + MAX_TRANS_COUNT + " ] ");

        Map<String, String> command = new HashMap<>();
        command.put("hashList", JSONObject.toJSONString(hashList));

        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_LIST);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 6000);
        transList = transListByResultList(rpcResultList);

        return transList;
    }

    /**
     * Description: 根据key维度查询交易
     *
     * @param key
     * @return Trans
     */
    public Trans getNewTransByKey(String key) {
        if (StringUtils.isBlank(key))
            return null;

        key = key.trim();
        if (SQLCheckUtil.checkSQLError(key)) //可能存在sql注入的key
            return null;

        Map<String, String> command = new HashMap<>();
        command.put("key", key);
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_KEY);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 3000);

        return transByResult(rpcResultList);
    }

    /**
     * Description:根据交易类type查询交易
     *
     * @param type
     * @return List<Trans>
     */
    public List<Trans> getTransListByType(String type) {
        if (StringUtils.isBlank(type))
            return new ArrayList<>();

        if (HashCheckUtil.illegalCharacterCheck(type) || type == null || type.length() > 45) {
            throw new TransInfoException("type have Illegal character or length too long.");
        }

        List<Trans> transList = null;
        Map<String, String> command = new HashMap<>();
        command.put("type", type);
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_LIST_BY_TYPE);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 6000);
        transList = transListByResultList(rpcResultList);

        return transList;
    }

    /**
     * Description:根据key维度查询交易历史
     *
     * @param key
     * @param startIndex
     * @param endIndex
     * @return List<Trans>
     */
    public List<Trans> getTransHistoryByKey(String key, int startIndex, int endIndex) {
        if (StringUtils.isBlank(key))
            return new ArrayList<>();

        List<Trans> transList = null;
        //验证参数  最多30条记录
        if (startIndex < 0 || startIndex > endIndex || ((endIndex - startIndex) >= MAX_TRANS_HISTORY_COUNT))
            throw new ParameterException("startIndex < 0 || startIndex > endIndex || ((endIndex - startIndex) >= " + MAX_TRANS_HISTORY_COUNT + ")");

        if (SQLCheckUtil.checkSQLError(key))
            return new ArrayList<>();

        Map<String, String> command = new HashMap<String, String>();
        command.put("key", key);
        command.put("startIndex", startIndex + "");
        command.put("endIndex", endIndex + "");

        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setCommand(command);
        msg.setTargetType(RPCMessage.TargetType.GET_TRANS_HISTORY);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 8000);
        transList = transListByResultList(rpcResultList);

        return transList;
    }


    /**
     * Description:获取当前区块链的连接数
     *
     * @return int
     */
    public int getConnectionCount() {
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());//交易hash标识唯一消息
        msg.setTargetType(RPCMessage.TargetType.GET_CONNECTION_COUNT);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 8000);

        int count = connectionCountByResultList(rpcResultList);

        return count;
    }

    /**
     * Description:开启事务
     *
     * @param keys
     * @return boolean
     */
    public boolean startTransaction(String[] keys) {
        if (keys != null && keys.length > 0) {
            //获取事务对象
            Transaction transaction = new Transaction(keys);
            //将事务提交到事务池，如果成功当前现在则锁住所有关于key的添加操作，在事务池中最多保留事务对象到stop time时间
            return ManagerTransactionPool.register(transaction, 6000);
        } else {
            return false;
        }
    }

    /**
     * Description:关闭事务
     *
     * @param keys
     */
    public void stopTransaction(String[] keys) {
        if (keys != null && keys.length > 0) {
            //获取事务对象
            Transaction transaction = new Transaction(keys);
            ManagerTransactionPool.destroy(transaction);
        }
    }

    /**
     * Description:查询交易总数量
     *
     * @return Long
     */
    public Long getTotalTransCount() {
        Long totalTransCount = 0L;
        RPCMessage msg = getMessage();
        msg.setMessageId(UUID.randomUUID().toString());
        msg.setTargetType(RPCMessage.TargetType.GET_TOTAL_TRANS_COUNT);

        net.request(msg);
        List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), 5000);
        if (rpcResultList == null || rpcResultList.size() == 0)
            return totalTransCount;

        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;

            if (result.getType() == RPCResult.ResultType.result_list) {
                try {
                    totalTransCount = JSONObject.parseObject(result.getEntity(), Long.class);
                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        }

        return totalTransCount;
    }

    /**
     * Description: 获取当前系统全部节点的node对象集合
     *
     * @return List<Node>
     */
    public List<Node> getBlockChainNodeStatus() {
        List<Node> nodes = net.getNodes();
        return nodes;
    }
    
    /**
     * @return String 获取当前connection对象的id
     */
    public String getId() {
		return this.connectionId;
	}
    

    /**
     * Description: 客户端构造器 ， 默认超时为 DEFAULT_TIMEOUT
     *
     * @param ipArr
     * @param port
     * @param token
     * @param ksPath
     * @param ksPasswd
     */
    protected Connection(String[] ipArr, int port, String token, String ksPath, String ksPasswd) {
        this(ipArr, port, token, DEFAULT_TIMEOUT, ksPath, ksPasswd);
    }

    /**
     * @param ipTable
     * @param port
     * @param key
     * @param token
     * @param cipher
     */
    protected Connection(String[] ipTable, int port, Key key, String token, Cipher cipher, NetType netType) {
        this.key = key;
        this.cipher = cipher;
        this.netType = netType;

        /**
         * @Description: 根据初始化的iptable长度计算最小吻合指数。
         */
        this.minSucces = PBFT.getMinByCount(ipTable.length);

        //开启net网络
        openNet(ipTable, port, token, cipher);

        //异步定时同步在线nodes
        asynAskNodes(DEFAULT_TIMEOUT);
    }

    /**
     * @param ipTable
     * @param port
     * @param key
     * @param token
     * @param cipher
     */
    protected Connection(String[] ipTable, int port, Key key, String token, Cipher cipher) {
        this(ipTable, port, key, token, cipher, NetType.NIO);
    }

    /**
     * DEFAULT_TIMEOUT
     *
     * @param ipArr
     * @param port
     * @param token
     * @param timeout
     * @param ksPath
     * @param ksPasswd
     */
    protected Connection(String[] ipArr, int port, String token, long timeout, String ksPath, String ksPasswd) {
        this(ipArr, port, token, timeout, ksPath, ksPasswd, NetType.NIO);
    }

    /**
     * DEFAULT_TIMEOUT
     *
     * @param ipArr
     * @param port
     * @param token
     * @param timeout
     * @param ksPath
     * @param ksPasswd
     * @param netType
     */
    protected Connection(String[] ipArr, int port, String token, long timeout, String ksPath, String ksPasswd, NetType netType) {
        this.netType = netType;

        this.minSucces = PBFT.getMinByCount(ipArr.length);

        //# 准备证书
        readyCert(ksPath, ksPasswd);

        //# 开启网络
        openNet(ipArr, port, token);

        //# 同步节点
        asynAskNodes(timeout);
    }

    protected RPCMessage getMessage() {
        RPCMessage msg = new RPCMessage();
        msg.setSender(this.connectionId);
        return msg;
    }


    private int connectionCountByResultList(List<RPCResult> rpcResultList) {
        if (CollectionUtils.isEmpty(rpcResultList)) {
            return 0;
        }
        List<Integer> list = new ArrayList<>();
        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;
            if (result.getType() == RPCResult.ResultType.connection_count) {
                try {
                    //反序列化可能异常
                    int count = Integer.parseInt(result.getEntity());
                    list.add(count);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        // 每个节点的连接数是异步刷新,未必完全一致
        if (list.size() >= this.minSucces || list.size() >= rpcResultList.size() / 2) {
            Map<Integer, Long> collect = list.stream().collect(Collectors.groupingBy(i -> i, Collectors.counting()));
            //{1,1,1,2} max = 3 || {1,2,1,2} max =2
            Long max = Collections.max(collect.values());
            //{1,1,1,2} count = 1 || {1,2,1,2} count =2
            List<Map.Entry<Integer, Long>> sameList = collect.entrySet().stream().filter(en -> en.getValue().longValue() == max.longValue()).collect(Collectors.toList());
            if (sameList.size() >= list.size() / 2) { // 取平均值
                int sum = list.stream().mapToInt(t -> t).sum();
                return sum / list.size();
            } else {
                return sameList.get(0).getKey();
            }
        }
        return 0;
    }


    private Trans transByResult(List<RPCResult> rpcResultList) {
        if (rpcResultList == null || rpcResultList.size() == 0)
            return null;

        int succesCount = 0;
        int failCount = 0;
        Trans succesTrans = null;
        Trans failTrans = null;
        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;

            Trans tempTrans = null;
            if (result.getType() == RPCResult.ResultType.tx_status) {
                try {
                    //反序列化可能异常
                    tempTrans = JSONObject.parseObject(result.getEntity(), Trans.class);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (tempTrans != null && TransUtil.getTransHash(tempTrans).equals(tempTrans.getHash())) {
                    //hash验证
                    succesCount++;//成功一次
                    succesTrans = tempTrans;//记住成功的交易对象
                } else {
                    failCount++;//失败一次
                    failTrans = tempTrans;//记住失败的交易对象
                }
            }
        }

        if (succesCount >= this.minSucces || succesCount > failCount) {
            return succesTrans;//返回成功的交易对象
        } else {
            return failTrans;
        }
    }

    private List<Trans> transListByResultList(List<RPCResult> rpcResultList) {
        List<Trans> transList = null;
        for (RPCResult result : rpcResultList) {
            if (stopNext(result))
                continue;

            if (result.getType() == RPCResult.ResultType.tx_list) {
                try {
                    transList = JSON.parseArray(result.getEntity(), Trans.class);
                    if (transList != null)
                        break;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        return transList;
    }

    /**
     * Description: 是否停止往下执行
     *
     * @param result
     * @return boolean
     */
    private boolean stopNext(RPCResult result) {
        if (result == null || StringUtils.isBlank(result.getEntity()))
            return true;
        return false;
    }

    /**
     * Description:开启网络连接
     *
     * @param ipArr
     * @param port
     * @param token
     */
    protected void openNet(String[] ipArr, int port, String token) {
        // 验证token不能为空
        if (StringUtils.isBlank(token)) {
            throw new ParameterException("token is empty ");
        }
        if (netType == null) {
            netType = NetType.NIO;
        }
        // 开启io
        switch (netType) {
            case IO:
                net = new IONet(ipArr, port, cipher, token, key, connectionId);
                break;
            case NIO:
                net = new NioNet(ipArr, port, cipher, token, key, connectionId);
                break;
            default:
                net = new NioNet(ipArr, port, cipher, token, key, connectionId);
                break;
        }
        net.start();
        Long start = System.currentTimeMillis();
        while (true) {
            if (net.getTaskSize() >= net.getMinNodeSize()) {
                break;
            }
            if ((System.currentTimeMillis() - start) >= 20000) {
                // 连接超时
                net.stop();
                throw new ConnectionTimeOutException("Connection time out, iptables=" + StringUtils.join(ipArr, ", ") + ",port=" + port + ",token=" + token);
            }
            try {
                Thread.sleep(20);
            } catch (InterruptedException e) {
            }

        }
    }

    protected void openNet(String[] ipArr, int port, String token, Cipher cipher) {
        this.cipher = cipher;
        openNet(ipArr, port, token);
    }

    /**
     * Description:异步同步节点信息
     *
     * @param timeout
     */
    protected void asynAskNodes(long timeout) {
        //# 每隔3秒同步一次
        scheduledService.scheduleAtFixedRate(() -> {
            RPCMessage msg = getMessage();
            msg.setTargetType(RPCMessage.TargetType.GET_NODE_LIST);
            msg.setMessageId(UUID.randomUUID().toString());
            //异步提交请求
            net.request(msg);
            //resphonse
            List<Node> nodes = new ArrayList<>();
            List<RPCResult> rpcResultList = net.resphone(msg.getMessageId(), timeout);
            for (RPCResult result : rpcResultList) {
                if (stopNext(result))
                    continue;

                if (result.getType() == RPCResult.ResultType.node_list) {
                    List<Node> nodeList = JSONObject.parseArray(result.getEntity(), Node.class);
                    if (nodeList != null && nodeList.size() > 0) {
                        if (nodeList.size() > nodes.size()) {
                            nodes = nodeList;
                        }
                    }
                }
            }

            //获取在线的节点添加到net中
            if (nodes != null && nodes.size() > 0) {
                for (Node node : nodes) {
                    if (SoutUtil.isOpenSout())
                        System.out.println("copy node id:" + node.getId() + " status=" + node.getStatus());
                    net.addNodeToNodes(node);
                }
            }

            try {
                Thread.sleep(1); //# 释放权限
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }, 0, 3, TimeUnit.SECONDS);

    }

    /**
     * Description:准备证书
     *
     * @param ksPath
     * @param ksPasswd
     */
    protected void readyCert(String ksPath, String ksPasswd) {
        try {
            String alias = Msp.ORGANIZATION_ALIAS;
            // 读取key store
            KeyStore keyStore = Msp.getKeyStore(ksPath, ksPasswd);

            String privateKey = cipher.getPrivateKeyStringByKeyStore(keyStore, ksPasswd, alias);
            String publicKey = cipher.getPublicKeyStringByStore(keyStore, ksPasswd, alias);

            //获取本地证书,该证书必须时由root证书颁发，否则无法与server建立连接。
            X509Certificate cert = (X509Certificate) keyStore.getCertificate(alias);
            String certBase64String = Msp.certToBase64String(cert);

            key.setPrivateKey(privateKey); //缓存私钥字符串
            key.setPublicKey(publicKey); // 缓存公钥字符串
            key.setLocalCertBase64String(certBase64String); //缓存证书base64字符串
        } catch (Exception e) {
            throw new CipherException("get private key by key store error:" + e.getMessage());
        }
    }

    private int minSucces = 1;
    protected Key key = new Key();
    protected Net net;
    protected NetType netType = NetType.NIO; //默认nio开启项目
    private String connectionId = UUID.randomUUID().toString();
    protected Cipher cipher = new Cipher();

    protected final static ScheduledExecutorService scheduledService = Executors.newSingleThreadScheduledExecutor();

}

