package cn.tdchain.api.rpc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.alibaba.fastjson.TypeReference;

import cn.tdchain.Trans;
import cn.tdchain.TransHead;
import cn.tdchain.api.config.SystemConfig;
import cn.tdchain.cb.constant.Commons;
import cn.tdchain.cb.constant.KeyAndType;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.util.CollectionUtils;
import cn.tdchain.cb.util.ConnectionUtils;
import cn.tdchain.cb.util.JsonUtils;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.cb.util.TdcbConfig;
import cn.tdchain.jbcc.BatchTrans;
import cn.tdchain.jbcc.Connection;
import cn.tdchain.jbcc.DateUtils;
import cn.tdchain.jbcc.PBFT;
import cn.tdchain.jbcc.ParameterException;
import cn.tdchain.jbcc.Result;
import cn.tdchain.jbcc.net.Net;
import cn.tdchain.jbcc.net.info.Node;
import cn.tdchain.jbcc.rpc.RPCBatchResult;
import cn.tdchain.jbcc.rpc.RPCMessage;

/**
 * TDCE connection.
 *
 * @version 1.0
 * @author Homer.J 2019-02-17
 */
public class CeConnection {

    private static final Logger log = LogManager.getLogger("TD_API");
    private static final ScheduledExecutorService SCHEDULED_SERVICE = Executors
            .newSingleThreadScheduledExecutor();
    private static final int DEFAULT_TIMEOUT = 3000;
    private String connectionId = UUID.randomUUID().toString(); // 每个连接器都又一个唯一标识
    private Net net;
    private Connection con = ConnectionUtils.getConnection();
    private int minSucces = 1;

    /**
     * Constructor.
     */
    public CeConnection() {
        // 验证token不能为空
        String token = TdcbConfig.getInstance().getConnectionToken();
        if (token == null || token.length() == 0) {
            throw new ParameterException("token is null");
        }
        this.minSucces = PBFT.getMinByCount(
                SystemConfig.getInstance().getCeIpTables().length);

        // 开启net网络
        net = new CeNioNet(SystemConfig.getInstance().getCeIpTables(),
                TdcbConfig.getInstance().getCePort(),
                TdcbConfig.getInstance().getCipher(), token,
                TdcbConfig.getInstance().getKey(), this.connectionId);
        log.info("Start net.");
        log.info("Try to connect nodes: {}",
                JsonUtils.toJson(SystemConfig.getInstance().getCeIpTables()));
        net.start();

        while (true) {
            if (net.getTaskSize() >= SystemConfig.getInstance()
                    .getCeIpTables().length) {
                log.info("Complete to connect {} CE nodes.", net.getTaskSize());
                break;
            } else {
                try {
                    Thread.sleep(20);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }

        asynAskNodes();
    }

    /**
     * Get JBCC connection.
     * 
     * @return JBCC connection
     */
    public Connection getCon() {
        return con;
    }

    /**
     * 异步同步节点信息.
     *
     * @param timeout
     */
    private void asynAskNodes() {
        // # 每隔3秒同步一次
        log.info("Start to sync CE nodes.");
        SCHEDULED_SERVICE.scheduleAtFixedRate(() -> {
            try {
                RPCMessage msg = getMessage();
                msg.setTargetType(RPCMessage.TargetType.REQUEST_NODE);
                msg.setMessageId(UUID.randomUUID().toString());
                // 异步提交请求
                net.request(msg);
                log.info("[{}] request node.", msg.getMessageId());
                // response
                RPCBatchResult batchResult = net.resphone(msg.getMessageId(),
                        DEFAULT_TIMEOUT);
                if (batchResult.isFail()) {
                    batchResult.getResult(); // 抛出异常？
                    return;
                }
                List<Result<Node>> rpcResultList = batchResult
                        .buildList(new TypeReference<Node>() {
                        });
                for (Result<Node> result : rpcResultList) {
                    if (result == null || !result.isSuccess()
                            || result.getEntity() == null) {
                        continue;
                    }

                    Node node = result.getEntity();
                    if (node != null) {
                        // 收到一个node
                        log.info("copy ce node id: {}, status={}", node.getId(),
                                node.getStatus());
                        net.addNodeToNodes(node);

                    }
                }
                Thread.sleep(1); // # 释放权限
            } catch (Exception e) {
                log.error("sync ce nodes get error:{}", e.getMessage());
            }
        }, 0, 3, TimeUnit.SECONDS);

    }

    /**
     * Get basic message.
     * 
     * @return RPCMessage with connectionId, messageID, time stamp
     */
    public RPCMessage getMessage() {
        RPCMessage msg = new RPCMessage();
        msg.setSender(this.connectionId); // 标识该消息来自哪个connection
        msg.setMessageId(UUID.randomUUID().toString()); // 交易hash标识唯一消息
        msg.setCommand(new HashMap<String, String>());
        msg.getCommand().put(Commons.TIMESTAMP,
                DateUtils.getCurrentTime().toString());
        return msg;
    }

    /**
     * Query contract.
     * 
     * @param msg RPCMessage
     * @return query result
     * @throws BusinessException business exception
     */
    public String queryAndReturn(RPCMessage msg) throws BusinessException {
        log.info("[{}] send query request.", msg.getTargetType().toString());
        net.request(msg);
        return responseQuery(msg);
    }

    private String responseQuery(RPCMessage msg) throws BusinessException {
        // response
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(),
                SystemConfig.getInstance().getTimeout());
        List<Result<String>> rpcResultList = batchResult
                .buildList(new TypeReference<String>() {
                });

        log.info("[{}] query response: {}", msg.getTargetType().toString(),
                rpcResultList.size());

        int failCount = 0;
        List<String> errorMessage = new ArrayList<String>();
        Map<String, Integer> count = new HashMap<String, Integer>();
        for (int i = 0; i < rpcResultList.size(); i++) {
            Result<String> rpcResult = rpcResultList.get(i);
            if (!rpcResult.isSuccess()) {
                if (!errorMessage.contains(rpcResult.getMsg())) {
                    errorMessage.add(rpcResult.getMsg());
                }
                failCount++;
                continue;
            }
            String entity = rpcResult.getEntity();
            if (StringUtils.isBlank(entity)) {
                if (!errorMessage.contains("Null.")) {
                    errorMessage.add("Null.");
                }
                failCount++;
                continue;
            }
            count.putIfAbsent(entity, 0);
            count.put(entity, count.get(entity) + 1);
        }
        if (failCount >= this.minSucces) {
            log.error("{} {} error msg: {}", ResultConstants.BFT_FAILED,
                    failCount, errorMessage.toString());
            throw new BusinessException(errorMessage.toString());
        }
        for (Map.Entry<String, Integer> entry : count.entrySet()) {
            if (entry.getValue().intValue() >= this.minSucces) {
                return entry.getKey();
            }
        }
        log.error("{} error msg: {}", ResultConstants.BFT_FAILED,
                errorMessage.toString());
        log.error("========================");
        log.error(JsonUtils.toJson(count));
        log.error("========================");
        throw new BusinessException(ResultConstants.BFT_FAILED);
    }

    /**
     * Request and response.
     * 
     * @param msg RPCMessage
     * @param failedMessage failed message prefix
     * @return response
     * @throws BusinessException business exception
     */
    public Map<String, String> sendAndReturn(RPCMessage msg,
                                             String failedMessage)
        throws BusinessException {
        log.info("[{}][{}] send request.", msg.getMessageId(),
                msg.getTargetType().toString());
        log.info("Dest nodes: ");
        log.info(net.getNodes().stream().map(Node::getId)
                .collect(Collectors.joining(";")));
        net.request(msg);
        return response(msg, failedMessage);
    }

    private Map<String, String> response(RPCMessage msg, String failedKey)
        throws BusinessException {
        // response
        RPCBatchResult batchResult = net.resphone(msg.getMessageId(),
                SystemConfig.getInstance().getTimeout());
        List<Result<List<Trans>>> rpcResultList = batchResult
                .buildList(new TypeReference<List<Trans>>() {
                });

        log.info("[{}][{}] ceconnection response: {}", msg.getMessageId(),
                msg.getTargetType().toString(), rpcResultList.size());

        int failCount = 0;
        Map<String, List<String>> errMsg = new HashMap<String, List<String>>();
        Map<Integer, Integer> count = new HashMap<Integer, Integer>();
        Map<Integer, List<Trans>> sta = new HashMap<Integer, List<Trans>>();
        for (int i = 0; i < rpcResultList.size(); i++) {
            Result<List<Trans>> rpcResult = rpcResultList.get(i);
            if (!rpcResult.isSuccess()) {
                errMsg.putIfAbsent(rpcResult.getMsg(), new ArrayList<String>());
                errMsg.get(rpcResult.getMsg()).add(String.valueOf(i));
                failCount++;
                continue;
            }
            if (CollectionUtils.isEmpty(rpcResult.getEntity())) {
                errMsg.putIfAbsent("Null.", new ArrayList<String>());
                errMsg.get("Null.").add(String.valueOf(i));
                failCount++;
                continue;
            }
//            List<Trans> tmp = JsonUtils.fromJson(rpcResult.getEntity(),
//                    new TypeReference<List<Trans>>() {
//                    }.getType());
            statistic(i, rpcResult.getEntity(), count, sta);
        }

        // 大部分节点运行没通过
        if (failCount >= this.minSucces) {
            log.error("======================== [{}][{}] FAILED",
                    msg.getMessageId(), msg.getTargetType().toString());
            for (Map.Entry<String, List<String>> entry : errMsg.entrySet()) {
                log.error(entry.getKey() + ":");
                log.error(JsonUtils.toJson(entry.getValue()));
            }
            log.error("========================");
            throw new BusinessException(ResultConstants.getFailedMsg(failedKey,
                    errMsg.keySet().toString()));
        }

        List<Trans> txs = getPbft(count, sta);
        // 节点结果不一致
        if (CollectionUtils.isEmpty(txs)) {
            log.error("======================== [{}][{}] FAILED",
                    msg.getMessageId(), msg.getTargetType().toString());
            if (CollectionUtils.isNotEmpty(errMsg)) {
                log.error(JsonUtils.toJson(errMsg));
            }
            if (CollectionUtils.isNotEmpty(count)) {
                log.error(JsonUtils.toJson(count));
            }
            if (CollectionUtils.isNotEmpty(sta)) {
                log.error(JsonUtils.toJson(sta));
            }
            log.error("========================");
            throw new BusinessException(ResultConstants.BFT_FAILED);
        }

        log.debug("******************************");
        log.debug(JsonUtils.toJson(txs));
        log.debug("******************************");
        BatchTrans<Trans> batch = new BatchTrans<>();
        batch.addTransToBatch(txs);
        Result<BatchTrans<TransHead>> backBatch = null;
        if (con.startTransaction(batch.keyToArray())) {
            backBatch = con.addBatchTrans(batch);
            con.stopTransaction(batch.keyToArray());
        }
        // 返回值异常
        if (backBatch == null) {
            throw new BusinessException("Null result.");
        }
        // 超时
        if (backBatch.isTimeout()
                && CollectionUtils.isNotEmpty(backBatch.getHashs())) {
            Map<String, String> resultMap = new HashMap<String, String>();
            for (String hash : backBatch.getHashs()) {
                Result<Trans> tx = this.getTransByHash(hash);
                if (tx == null || !tx.isSuccess() || tx.getEntity() == null) {
                    throw new BusinessException(ResultConstants.ONCHAIN_FAILED);
                }
                String type = tx.getEntity().getType()
                        .split(KeyAndType.SPLIT)[0];
                resultMap.put(KeyAndType.getDescMap().get(type), hash);
            }
            return resultMap;
        }
        // 失败
        if (backBatch.isFail()) {
            String failedMsg = ResultConstants.getFailedMsg(failedKey,
                    backBatch.getMsg());
            log.error("======================== [{}][{}] FAILED",
                    msg.getMessageId(), msg.getTargetType().toString());
            log.error(failedMsg);
            throw new BusinessException(failedMsg);
        }

        // 成功
        if (backBatch.isSuccess() && backBatch.getEntity() != null) {
            Map<String, String> resultMap = new HashMap<String, String>();
            backBatch.getEntity().getTransSet().forEach(tx -> {
                String type = tx.getType().split(KeyAndType.SPLIT)[0];
                resultMap.put(KeyAndType.getDescMap().get(type), tx.getHash());
            });
            log.info("======================== [{}][{}] SUCCESS",
                    msg.getMessageId(), msg.getTargetType().toString());
            log.info(JsonUtils.toJson(resultMap));
            return resultMap;
        }
        throw new BusinessException(ResultConstants.ONCHAIN_FAILED);
    }

    private List<Trans> getPbft(Map<Integer, Integer> count,
                                Map<Integer, List<Trans>> sta) {
        Integer selectNo = null;
        for (Map.Entry<Integer, Integer> entry : count.entrySet()) {
            if (entry.getValue() >= this.minSucces) {
                selectNo = entry.getKey();
                break;
            }
        }
        if (selectNo != null) {
            return sta.get(selectNo);
        }
        return null;
    }

    private void statistic(int no, List<Trans> tmp, Map<Integer, Integer> count,
                           Map<Integer, List<Trans>> sta) {
        if (CollectionUtils.isEmpty(count)) {
            count.put(no, 1);
            sta.put(no, tmp);
            return;
        }
        boolean matchFlag = false;
        for (Map.Entry<Integer, List<Trans>> entry : sta.entrySet()) {
            Integer num = entry.getKey();
            List<Trans> staTrans = entry.getValue();
            if (staTrans.size() != tmp.size()) {
                continue;
            }
            boolean inconsistent = false;
            for (int i = 0; i < staTrans.size(); i++) {
                if (!staTrans.get(i).getHash().equals(tmp.get(i).getHash())) {
                    inconsistent = true;
                    break;
                }
            }
            if (inconsistent) {
                continue;
            }
            matchFlag = true;
            count.put(num, count.get(num) + 1);
            // 添加签名
            for (int i = 0; i < staTrans.size(); i++) {
                for (Map.Entry<String, String> tmpEntry : tmp.get(i)
                        .getSignMap().entrySet()) {
                    if (staTrans.get(i).getSignMap() == null) {
                        staTrans.get(i)
                                .setSignMap(new HashMap<String, String>());
                    }
                    staTrans.get(i).getSignMap().put(tmpEntry.getKey(),
                            tmpEntry.getValue());
                    break;
                }
            }
            break;
        }
        if (!matchFlag) {
            count.put(no, 1);
            sta.put(no, tmp);
        }

    }

    /**
     * Get latest transaction by connectionId and key.
     * 
     * @param key String
     * @return transaction
     */
    public Result<Trans> getNewTransByKey(String key) {
        return con.getNewTransByAccountAndKey(con.getAccount(), key);
    }

    /**
     * Get transaction list by type(up to 1000 records).
     * 
     * @param type String
     * @return transaction list
     */
    public Result<List<Trans>> getTransListByType(String type) {
        // FIXME:
        return con.getTransListByAccountAndType(con.getAccount(), type, 1000,
                0);
    }

    /**
     * Get transaction count by type.
     * 
     * @param type String
     * @return integer
     */
    public Result<Integer> getTransCountByType(String type) {
        return con.getTransCountByAccountAndType(con.getAccount(), type);
    }

    /**
     * Get transaction by hash.
     * 
     * @param hash String
     * @return Result
     */
    public Result<Trans> getTransByHash(String hash) {
        return con.getTransByAccountAndHash(con.getAccount(), hash);
    }

    /**
     * Query history by key(up to 20 records).
     *
     * @param key String
     * @return Result
     */
    public Result<List<Trans>> getTransHistoryByKey(String key) {
        return con.getTransHistoryByKey(key, 0, 19);
    }

}
