package cn.tdchain.api.service.impl;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.Trans;
import cn.tdchain.TransHead;
import cn.tdchain.api.config.SystemConfig;
import cn.tdchain.api.rpc.CeConnection;
import cn.tdchain.api.rpc.CeConnectionUtils;
import cn.tdchain.api.service.AccountService;
import cn.tdchain.cb.constant.ContractKey;
import cn.tdchain.cb.constant.KeyAndType;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.domain.Account;
import cn.tdchain.cb.domain.AccountContract;
import cn.tdchain.cb.domain.AccountTemplate;
import cn.tdchain.cb.domain.Auth;
import cn.tdchain.cb.domain.ContractState;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.service.impl.CipherServiceImpl;
import cn.tdchain.cb.util.AddressUtils;
import cn.tdchain.cb.util.Base64Util;
import cn.tdchain.cb.util.CollectionUtils;
import cn.tdchain.cb.util.IOUtils;
import cn.tdchain.cb.util.JsonUtils;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.cb.util.TdcbConfig;
import cn.tdchain.cb.util.TransUtils;
import cn.tdchain.cipher.Cipher;
import cn.tdchain.jbcc.DateUtils;
import cn.tdchain.jbcc.Result;

/**
 * Service Implementation for account.
 * 
 * @version 1.0
 * @author bingoer.H 2018-11-29
 */
public class AccountServiceImpl implements AccountService {

    private CipherServiceImpl cipherService = new CipherServiceImpl();

    private CeConnection con = CeConnectionUtils.getConnection();

    @Override
    public String createSuper() throws BusinessException {
        String superAddress = SystemConfig.getInstance().getSuperAddress();
        if (superAddress.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, superAddress));
        }
        Account superAccount = findLatest(superAddress);
        if (superAccount != null) {
            throw new BusinessException(ResultConstants.ACCOUNT_EXISTS);
        }

        String keyPath = AddressUtils.getKsFilePath(superAddress);

        String pubKey = cipherService.getPublicKeyStr(Cipher.Type.RSA.name(),
                keyPath, SystemConfig.getInstance().getSuperPassword());
        if (StringUtils.isBlank(pubKey)) {
            throw new BusinessException(ResultConstants.PUBLIC_KEY_FAILED);
        }
        String keyStoreStr = IOUtils.getBytes(keyPath);
        if (StringUtils.isBlank(keyStoreStr)) {
            throw new BusinessException(ResultConstants.KEYSTORE_FAILED);
        }

        Account newAccount = new Account(superAddress, Cipher.Type.RSA.name(),
                keyStoreStr, null, DateUtils.getCurrentTime());
        // 添加超级管理员权限
        newAccount.setAuth(Auth.SUPER);
        Trans tx = TransUtils.createTrans(newAccount, con.getCon().getAccount(),
                newAccount.getTimestamp());
        Result<TransHead> txResult = con.getCon().addTrans(tx);
        if (txResult != null && txResult.isSuccess()) {
            return superAddress;
        } else {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.ACCOUNT_CREATE_FAILED, txResult.getMsg()));
        }
    }

    @Override
    public String create(String ksPassword, String crypto, String info)
        throws BusinessException {
        if (StringUtils.isBlank(ksPassword) || StringUtils.isBlank(crypto)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        crypto = crypto.toUpperCase();
        String address = cipherService.generateKeyStore(crypto, ksPassword);
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.KEYSTORE_FAILED);
        }
        String keyPath = AddressUtils.getKsFilePath(address);

        String keyStoreStr = IOUtils.getBytes(keyPath);
        if (StringUtils.isBlank(keyStoreStr)) {
            throw new BusinessException(ResultConstants.KEYSTORE_FAILED);
        }

        String pubKey = cipherService.getPublicKeyStr(crypto, keyPath,
                ksPassword);
        if (StringUtils.isBlank(pubKey)) {
            throw new BusinessException(ResultConstants.PUBLIC_KEY_FAILED);
        }

        String encryptedInfo = null;
        if (!StringUtils.isBlank(info)) {
            encryptedInfo = cipherService.encrypt(crypto, info, pubKey);
            if (StringUtils.isBlank(encryptedInfo)) {
                throw new BusinessException(ResultConstants.USERINFO_EN_FAILED);
            }
        }
        Account account = new Account(address, crypto, keyStoreStr,
                encryptedInfo, DateUtils.getCurrentTime());
        Trans tx = TransUtils.createTrans(account, con.getCon().getAccount(),
                account.getTimestamp());
        Result<TransHead> txResult = con.getCon().addTrans(tx);
        if (txResult != null && txResult.isSuccess()) {
            return address;
        } else {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.ACCOUNT_CREATE_FAILED, txResult.getMsg()));
        }
    }

    @Override
    public String encryptInfo(String address, String ksPassword, String info)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(info)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        Result<Trans> latest = con
                .getNewTransByKey(KeyAndType.getAccountKey(address));

        if (latest == null) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }
        if (!latest.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (latest.getEntity() == null
                || StringUtils.isBlank(latest.getEntity().getData())) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }
        Account account = JsonUtils.fromJson(latest.getEntity().getData(),
                Account.class);
        if (account == null) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }
        if (account.isFrozen()) {
            throw new BusinessException(ResultConstants.ACCOUNT_FROZEN);
        }

        if (StringUtils.isBlank(account.getKeyStore())) {
            throw new BusinessException(ResultConstants.KEYSTORE_FAILED);
        }
        String filePath = AddressUtils.getKsFilePath(account.getAddress());
        File file = new File(filePath);
        if (!file.exists()) {
            IOUtils.generateFile(account.getKeyStore(),
                    TdcbConfig.getInstance().getAccountKsPath(), filePath);
        }
        String publicKey = cipherService.getPublicKeyStr(account.getCrypto(),
                filePath, ksPassword);

        if (StringUtils.isBlank(publicKey)) {
            throw new BusinessException(ResultConstants.PUBLIC_KEY_FAILED);
        }
        String keyStoreAddress = cipherService.getAddress(account.getCrypto(),
                publicKey);
        if (!account.getAddress().equals(keyStoreAddress)) {
            throw new BusinessException(ResultConstants.KS_NOT_MATCH);
        }

        String encryptedInfo = cipherService.encrypt(account.getCrypto(), info,
                publicKey);
        if (StringUtils.isBlank(encryptedInfo)) {
            throw new BusinessException(ResultConstants.USERINFO_EN_FAILED);
        }
        return encryptedInfo;
    }

    @Override
    public JSONObject findLatestWithoutKs(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }

        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        Result<Trans> tx = con
                .getNewTransByKey(KeyAndType.getAccountKey(address));
        if (tx == null) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }
        if (!tx.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (tx.getEntity() == null
                || StringUtils.isBlank(tx.getEntity().getData())) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }
        JSONObject temp = JSONObject.parseObject(tx.getEntity().getData());
        temp.put("hash", tx.getEntity().getHash());
        temp.remove("keyStore");
        return temp;
    }

    @Override
    public String findInfo(String address, String ksPassword)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        Account account = findLatest(address);
        if (account == null) {
            throw new BusinessException(ResultConstants.ACCOUNT_NOT_EXISTS);
        }

        if (StringUtils.isBlank(account.getInfo())) {
            return null;
        }
        String filePath = AddressUtils.getKsFilePath(account.getAddress());
        File file = new File(filePath);
        if (!file.exists()) {
            IOUtils.generateFile(account.getKeyStore(),
                    TdcbConfig.getInstance().getAccountKsPath(), filePath);
        }
        String privateKey = cipherService.getPrivateKeyStr(account.getCrypto(),
                filePath, ksPassword);
        if (StringUtils.isBlank(privateKey)) {
            // 密码错误则返回密文
            return account.getInfo();
        }
        return cipherService.decrypt(account.getCrypto(), account.getInfo(),
                privateKey);
    }

    @Override
    public List<JSONObject> queryHistory(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        List<JSONObject> accounts = new ArrayList<JSONObject>();

        Result<List<Trans>> alTxs = con
                .getTransHistoryByKey(KeyAndType.getAccountKey(address));
        if (alTxs == null) {
            return accounts;
        }
        if (!alTxs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (CollectionUtils.isEmpty(alTxs.getEntity())) {
            return accounts;
        }
        alTxs.getEntity().forEach(tx -> {
            JSONObject temp = JSONObject.parseObject(tx.getData());
            temp.put("hash", tx.getHash());
            temp.remove("keyStore");
            accounts.add(temp);
        });
        return accounts;
    }

    @Override
    public List<String> queryContractNames(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        List<String> contracts = new ArrayList<String>();

        Result<List<Trans>> txs = con
                .getTransListByType(KeyAndType.getAccountContractType(address));
        if (txs == null) {
            return contracts;
        }
        if (!txs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (txs.getEntity() == null
                || CollectionUtils.isEmpty(txs.getEntity())) {
            return contracts;
        }
        txs.getEntity().forEach(tx -> {
            if (tx.getData() != null) {
                AccountContract ac = JsonUtils.fromJson(tx.getData(),
                        AccountContract.class);
                if (ac != null) {
                    contracts.add(ac.getContractName());
                }
            }
        });
        return contracts;
    }

    private Account findLatest(String address) throws BusinessException {
        if (StringUtils.isBlank(address)) {
            return null;
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        try {
            Result<Trans> tx = con
                    .getNewTransByKey(KeyAndType.getAccountKey(address));
            if (tx == null) {
                return null;
            }
            if (!tx.isSuccess()) {
                throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
            }
            if (tx.getEntity() == null
                    || StringUtils.isBlank(tx.getEntity().getData())) {
                return null;
            }
            return JsonUtils.fromJson(tx.getEntity().getData(), Account.class);
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public List<String> queryTemplateNames(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        List<String> templates = new ArrayList<String>();
        Result<List<Trans>> txs = con
                .getTransListByType(KeyAndType.getAccountTemplateType(address));
        if (txs == null) {
            return templates;
        }
        if (!txs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (txs.getEntity() == null
                || CollectionUtils.isEmpty(txs.getEntity())) {
            return templates;
        }
        txs.getEntity().forEach(tx -> {
            if (tx.getData() != null) {
                AccountTemplate at = JsonUtils.fromJson(tx.getData(),
                        AccountTemplate.class);
                if (at != null) {
                    templates.add(at.getTemplateName());
                }
            }
        });
        return templates;
    }

    @Override
    public List<JSONObject> queryLedger(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        Result<List<Trans>> txs = con
                .getTransListByType(KeyAndType.getAccountLedgerType(address));

        List<JSONObject> ledger = new ArrayList<JSONObject>();
        if (txs == null) {
            return ledger;
        }
        if (!txs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (txs.getEntity() == null
                || CollectionUtils.isEmpty(txs.getEntity())) {
            return ledger;
        }
        txs.getEntity().forEach(tx -> {
            if (StringUtils.isNotBlank(tx.getData())) {
                JSONObject temp = JSONObject.parseObject(tx.getData());
                temp.put("hash", tx.getHash());
                temp.put("csHash",
                        temp.get("referenceHash") + tx.getHeight().toString());
                ledger.add(temp);
            }
        });
        return ledger;
    }

    @Override
    public List<JSONObject> queryLedgerHistory(String address,
                                               String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        List<JSONObject> history = new ArrayList<JSONObject>();
        Result<List<Trans>> alTxs = con.getTransHistoryByKey(
                KeyAndType.getAccountLedgerKey(address, contractName));
        if (alTxs == null) {
            return history;
        }
        if (!alTxs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (CollectionUtils.isEmpty(alTxs.getEntity())) {
            return history;
        }
        alTxs.getEntity().forEach(tx -> {
            if (StringUtils.isNotBlank(tx.getData())) {
                JSONObject temp = JSONObject.parseObject(tx.getData());
                temp.put("hash", tx.getHash());
                temp.put("csHash",
                        temp.get("referenceHash") + tx.getHeight().toString());
                history.add(temp);
            }
        });
        return history;
    }

    @Override
    public List<JSONObject> queryLedgerHistoryEx(String address,
                                                 String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }

        List<JSONObject> history = new ArrayList<JSONObject>();
        Result<List<Trans>> alTxs = con.getTransHistoryByKey(
                KeyAndType.getAccountLedgerKey(address, contractName));
        if (alTxs == null) {
            return history;
        }
        if (!alTxs.isSuccess()) {
            throw new BusinessException(ResultConstants.GETCHAIN_FAILED);
        }
        if (CollectionUtils.isEmpty(alTxs.getEntity())) {
            return history;
        }
        alTxs.getEntity().forEach(tx -> history.add(construct(tx)));
        return history;
    }

    private JSONObject construct(Trans latest) {
        JSONObject temp = JSONObject.parseObject(latest.getData());
        temp.put("hash", latest.getHash());
        String csHash = temp.getString("referenceHash") +latest.getHeight().toString();
        temp.put("csHash", csHash);
        Result<Trans> refCs = con.getTransByHash(csHash);
        if (refCs == null || !refCs.isSuccess() || refCs.getEntity() == null
                || StringUtils.isBlank(refCs.getEntity().getData())) {
            return temp;
        }
        ContractState cs = JsonUtils.fromJson(refCs.getEntity().getData(),
                ContractState.class);
        temp.put("operator", cs.getOperator());
        temp.put("operation", cs.getOperation());
        parseOpr(cs.getOprMap(), ContractKey.CREATE_ARGS);
        parseOpr(cs.getOprMap(), ContractKey.UPDATE_ARGS);
        parseOpr(cs.getOprMap(), ContractKey.UPDATE_SYNC_ARGS);
        parseOpr(cs.getOprMap(), ContractKey.RUN_ARGS);
        temp.put("oprMap", cs.getOprMap());
        return temp;
    }

    private void parseOpr(Map<String, Object> oprMap, String key) {
        if (oprMap.containsKey(key)) {
            oprMap.put(key, Base64Util.decoder((String) oprMap.get(key)));
        }
    }

    @Override
    public JSONObject getAccountStatistic(String address)
        throws BusinessException {
        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        address = address.trim();

        JSONObject result = new JSONObject();
        Result<Integer> als = con
                .getTransCountByType(KeyAndType.getAccountLedgerType(address));
        if (als.isSuccess()) {
            result.put("asset", als.getEntity());
        }

        Result<Integer> ats = con.getTransCountByType(
                KeyAndType.getAccountTemplateType(address));
        if (ats.isSuccess()) {
            result.put("template", ats.getEntity());
        }

        Result<Integer> acs = con.getTransCountByType(
                KeyAndType.getAccountContractType(address));
        if (acs.isSuccess()) {
            result.put("contract", acs.getEntity());
        }
        return result;
    }

}
