package cn.tdchain.api.rpc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import cn.tdchain.RPCResultJSONObject;
import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.Key;
import cn.tdchain.jbcc.PBFT;
import cn.tdchain.jbcc.Result;
import cn.tdchain.jbcc.net.info.Node;
import cn.tdchain.jbcc.net.nio.NioNet;
import cn.tdchain.jbcc.rpc.RPCBatchResult;
import cn.tdchain.jbcc.rpc.RPCMessage;
import cn.tdchain.jbcc.rpc.RPCResult;
import cn.tdchain.jbcc.rpc.nio.client.NioRpcClient;
import cn.tdchain.jbcc.rpc.nio.client.NioRpcSynResponseClient;
import cn.tdchain.jbcc.rpc.nio.handler.NioRpcChannelFactory;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.UnpooledByteBufAllocator;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.LengthFieldBasedFrameDecoder;
import io.netty.handler.codec.LengthFieldPrepender;
import io.netty.handler.codec.string.StringDecoder;
import io.netty.handler.codec.string.StringEncoder;
import io.netty.util.CharsetUtil;

/**
 * NIO版本的net实现.
 *
 * @author Homer.J 2019-02-17
 */
public class CeNioNet extends NioNet {

    private static final Logger log = LogManager.getLogger("TD_API");

    private final Bootstrap bootstrap;

    private EventLoopGroup workGroup;

    private boolean status = true;

    private HashSet<String> iptables;

    private Hashtable<String, Node> nodes = new Hashtable<>();

    private int serverPort;

    private Cipher cipher;

    private String token;

    private Key key;

    private String connectionId;

    private int minOnlineNodes = 1;

    private NioRpcChannelFactory nioRpcChannelFactory;

    private Map<String, NioTask> taskList = new ConcurrentHashMap<>();

    /**
     * Constructor.
     * 
     * @param iptables ce ip tables
     * @param serverPort ce port
     * @param cipher cipher
     * @param token token
     * @param key key
     * @param connectionId String
     */
    public CeNioNet(String[] iptables, int serverPort, Cipher cipher,
            String token, Key key, String connectionId) {
        super(iptables, serverPort, cipher, token, key, connectionId);
        this.iptables = new HashSet<>(Arrays.asList(iptables));
        this.serverPort = serverPort;
        this.cipher = cipher;
        this.token = token;
        this.key = key;
        this.connectionId = connectionId;

        bootstrap = new Bootstrap();
        workGroup = new NioEventLoopGroup();
        bootstrap.group(workGroup).channel(NioSocketChannel.class)
                .option(ChannelOption.TCP_NODELAY, true)
                .option(ChannelOption.SO_KEEPALIVE, true)
                .handler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch)
                        throws Exception {
                        ch.config()
                                .setAllocator(UnpooledByteBufAllocator.DEFAULT);
                        ch.pipeline()
                                .addLast(new LengthFieldBasedFrameDecoder(
                                        Integer.MAX_VALUE, 0, 4, 0, 4))
                                .addLast(new LengthFieldPrepender(4))
                                .addLast(new StringDecoder(CharsetUtil.UTF_8))
                                .addLast(new StringEncoder(CharsetUtil.UTF_8));
                    }
                });

        nioRpcChannelFactory = new NioRpcChannelFactory(bootstrap);
    }

    @Override
    public void start() {
        asynGetNodesByIpTable();
        // 动态运行每个节点的Task
        new Thread(() -> {
            while (true) {
                if (!status) {
                    taskList.forEach((k, v) -> {
                        v.stop();
                    });
                    break;
                }

                try {
                    log.info("ce node size=" + nodes.size());
                    log.info("ce task size=" + taskList.size());
                    nodes.forEach((id, node) -> {
                        startNode(id, node);
                    });

                    // 休息
                    Thread.sleep(2000);
                } catch (Exception e) {
                }

            }

        }).start();
    }

    private void startNode(String id, Node node) {
        log.info("serverip={}, status={}", node.serverIP(), node.getStatus());
        NioTask t = taskList.get(id);
        if (t != null && !t.isStatus()) {
            t.stop();
            taskList.remove(id);
            t = null;
        }

        if (!Node.NodeStatus.DIE.equals(node.getStatus())
                && !Node.NodeStatus.OUT.equals(node.getStatus())) {
            if (t == null) {
                if (taskList.size() < iptables.size()) {
                    try {
                        t = new NioTask(node.serverIP(), serverPort, cipher,
                                token, key, connectionId, 1);
                        t.start();
                        taskList.put(id, t);
                    } catch (Exception e) {
                        t = null;
                        log.error("Failed to connect ce node: "
                                + node.serverIP() + " " + e.getMessage(), e);
                        return;
                    }
                }
            }
        } else {
            if (t != null && t.isStatus()) { // 如果节点die，task还在连接，则任务需要销毁
                t.stop();
                taskList.remove(id);
            }

            log.warn("dead node id=" + node.getId());
        }
        // 可能网络原因、或者节点下线造成通信失败的，该任务需要销毁。
        if (t != null && !t.isStatus()) {
            t.stop();
            taskList.remove(id);
        }

    }

    @Override
    public void request(RPCMessage msg) {
        taskList.forEach((k, v) -> {
            if (v != null && v.isStatus()) {
                msg.setTarget(k);
                v.addRequest(msg.clone());
            } else {
                log.error("Request error {}.", k);
            }

        });

    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    @Override
    public RPCBatchResult resphone(String messageId, long timeOut) {
        RPCBatchResult rpcBatchResult = RPCBatchResult.newInstance();
        long start = System.currentTimeMillis();

        log.info("[" + messageId + "] response ce task list: "
                + taskList.size());
        List<NioTask> tasks = new ArrayList<>(taskList.size() + 3);
        taskList.forEach((k, v) -> {
            if (v != null) {
                tasks.add(v);
            }
        });
        log.info("[" + messageId + "] response ce tasks: " + tasks.size());
        while (true) {
            tasks.forEach((v) -> {
                if (v != null) {
                    RPCResult r = v.poll(messageId); // 弹出消息，没有则返回空

                    if (r != null) {
                        Result result = new Result();
                        result.setStatus(r.getStatus());
                        result.setEntity(r.getEntity());
                        result.setMsg(r.getMsg());
                        rpcBatchResult.add(result);
                    }
                }
            });

            if ((rpcBatchResult.size() >= iptables.size())) {
                break; // 结果集达到要求或者超时时间到达就退出while循环
            }

            if ((System.currentTimeMillis() - start) > timeOut) {
                if (rpcBatchResult.size() >= minOnlineNodes) {
                    log.info("[{}] ce result size = {}", this.serverPort,
                            rpcBatchResult.size());
                    break;
                }
                rpcBatchResult.isTimeOut(true);
                return rpcBatchResult;
            }

            try {
                Thread.sleep(20);
            } catch (InterruptedException e) {
            }
        }

        return rpcBatchResult;
    }

    @Override
    public int getTaskSize() {
        synchronized (taskList) {
            return taskList.size();
        }
    }

    @Override
    public int getMinNodeSize() {
        minOnlineNodes = 0;
        this.nodes.forEach((k, v) -> {
            if (v.getStatus() == Node.NodeStatus.METRONOMER) {
                minOnlineNodes++;
            }
        });
        if (minOnlineNodes < this.iptables.size()) {
            minOnlineNodes = this.iptables.size();
        }

        return PBFT.getMinByCount(minOnlineNodes);
    }

    @Override
    public void addNodeToNodes(Node node) {
        // 只是新加新的node对象
        if (this.nodes.get(node.getId()) == null) {
            this.nodes.put(node.getId(), node);
        } else {
            Node n = this.nodes.get(node.getId());
            n.setStatus(node.getStatus());
        }
    }

    private void asynGetNodesByIpTable() {
        new Thread(() -> {
            // 初始化
            HashMap<String, Boolean> flags = new HashMap<String, Boolean>();
            iptables.forEach(ip -> {
                if (ip != null && ip.length() > 0) {
                    flags.put(ip, false);
                }
            });
            HashMap<String, NioRpcSynResponseClient> clients = new HashMap<String, NioRpcSynResponseClient>();

            while (true) {
                if (!status) {
                    break;
                }
                // 1秒后开始
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e1) {
                }

                flags.forEach((ip, flag) -> {
                    try {
                        if (flag) {
                            return;
                        }
                        NioRpcSynResponseClient client = clients.get(ip);
                        if (client == null) {
                            client = new NioRpcSynResponseClient(
                                    new NioRpcClient(nioRpcChannelFactory, ip,
                                            serverPort, 3000, token,
                                            connectionId,
                                            key.getLocalCertBase64String()));
                            if (client.isActive()) {
                                clients.put(ip, client);
                            } else {
                                client.close();
                                client = null;
                                return;
                            }
                        }

                        // 封装请求休息体
                        RPCMessage requestNodeMsg = new RPCMessage();
                        requestNodeMsg.setTarget(ip);
                        requestNodeMsg.setTargetType(
                                RPCMessage.TargetType.REQUEST_NODE);
                        requestNodeMsg.setSender(connectionId);

                        String resultStr = client.sendAndReturn(
                                requestNodeMsg.toJsonString(), 5000);
                        if (resultStr != null) {
                            RPCResult<Node> r = RPCResultJSONObject
                                    .parseObject(resultStr, Node.class);
                            if (r != null && r.getEntity() != null) {
                                Node node = r.getEntity();
                                // 收到一个node
                                log.info("初次收到node=" + node.toJSONString());
                                nodes.put(node.getId(), node);

                                // 关闭连接资源
                                client.close();
                                clients.remove(ip);

                                // 标记已经收到一个node
                                flags.put(ip, true);

                            }
                        }

                    } catch (Exception e) {
                        log.error("request error server ip=" + ip, e);
                    }
                });

                Long count = flags.values().stream().filter(value -> value)
                        .count();

                if (count.intValue() >= iptables.size()) {
                    break;
                }
            }
        }).start();
    }

    @Override
    public List<Node> getNodes() {
        return this.nodes.entrySet().stream()
                .filter(en -> en.getValue() != null).map(en -> en.getValue())
                .collect(Collectors.toList());
    }

    @Override
    public void stop() {
        this.status = false;
        workGroup.shutdownGracefully();
    }

}
