package cn.tdchain.api;

import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.api.config.SystemConfig;
import cn.tdchain.api.rpc.CeConnection;
import cn.tdchain.api.rpc.CeConnectionUtils;
import cn.tdchain.api.service.AccountService;
import cn.tdchain.api.service.impl.AccountServiceImpl;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.jbcc.rpc.RPCMessage;

/**
 * Account API.
 * 
 * @version 1.0
 * @author Bingoer.H 2018-11-29
 */
public class AccountApi {

    private AccountService accountService = new AccountServiceImpl();
    private CeConnection con = CeConnectionUtils.getConnection();

    /**
     * Override super address and password.
     * 
     * @param address String
     * @param password String
     * @throws BusinessException business exception
     */
    public void overrideSuper(String address, String password)
        throws BusinessException {
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        SystemConfig.getInstance().overrideSuper(address, password);
    }

    /**
     * Create administrator(super) account.
     * 
     * @return account address and hash
     * @throws BusinessException business exception
     */
    public Map<String, String> createSuperAccount() throws BusinessException {
        return accountService.createSuper();
    }

    /**
     * Create account.
     * 
     * @param ksPassword key store password
     * @param crypto RSA
     * @param info 用户个人敏感信息 Map对象的json序列化字符串，链上加密存储
     * @return account address and hash
     * @throws BusinessException business exception
     */
    public Map<String, String> create(String ksPassword, String crypto, String info)
        throws BusinessException {
        return accountService.create(ksPassword, crypto, info);
    }

    /**
     * Freeze account.
     * 
     * @param address super account address
     * @param ksPassword super account key store password
     * @param freezeAddr address to be frozen
     * @return hashes
     * @throws BusinessException business exception
     */
    public Map<String, String> freeze(String address, String ksPassword,
                                      String freezeAddr)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(freezeAddr)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        if (freezeAddr.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, freezeAddr));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("address", address);
        msg.getCommand().put("ksPassword", ksPassword);
        msg.getCommand().put("frozenAddress", freezeAddr);
        msg.setTargetType(RPCMessage.TargetType.FREEZE_ACCOUNT);
        return con.sendAndReturn(msg, ResultConstants.ACCOUNT_FREEZE_FAILED);
    }

    /**
     * Unfreeze account.
     * 
     * @param address super account address
     * @param ksPassword super account key store password
     * @param unfreezeAddr address to be unfreezed
     * @return hashes
     * @throws BusinessException business exception
     */
    public Map<String, String> unfreeze(String address, String ksPassword,
                                        String unfreezeAddr)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(unfreezeAddr)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        if (unfreezeAddr.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, unfreezeAddr));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("address", address);
        msg.getCommand().put("ksPassword", ksPassword);
        msg.getCommand().put("unfreezeAddr", unfreezeAddr);
        msg.setTargetType(RPCMessage.TargetType.UNFREEZE_ACCOUNT);
        return con.sendAndReturn(msg, ResultConstants.ACCOUNT_UNFREEZE_FAILED);
    }

    /**
     * Update account's user info (store after encryption).
     * 
     * @param address account address
     * @param ksPassword key store password
     * @param userInfo user info
     * @return hashes
     * @throws BusinessException business exception
     */
    public Map<String, String> updateInfo(String address, String ksPassword,
                                          String userInfo)
        throws BusinessException {
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        String encryptInfo = accountService.encryptInfo(address, ksPassword,
                userInfo);
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("address", address);
        msg.getCommand().put("ksPassword", ksPassword);
        msg.getCommand().put("userInfo", userInfo);
        msg.getCommand().put("encryptInfo", encryptInfo);
        msg.setTargetType(RPCMessage.TargetType.UPDATE_INFO);
        return con.sendAndReturn(msg, ResultConstants.ACCOUNT_UPDATE_FAILED);
    }

    /**
     * Update authority.
     * 
     * @param address operator address
     * @param ksPassword operator key store password
     * @param modifiedAddr account address to be modified
     * @param authStr update authority, SUPER not allowed
     * @return hashes
     * @throws BusinessException business exception
     */
    public Map<String, String> updateAuth(String address, String ksPassword,
                                          String modifiedAddr, String authStr)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(modifiedAddr)
                || StringUtils.isBlank(authStr)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        if (modifiedAddr.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, modifiedAddr));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("address", address);
        msg.getCommand().put("ksPassword", ksPassword);
        msg.getCommand().put("modifiedAddr", modifiedAddr);
        msg.getCommand().put("authStr", authStr);
        msg.setTargetType(RPCMessage.TargetType.UPDATE_AUTH);
        return con.sendAndReturn(msg, ResultConstants.AUTH_UPDATE_FAILED);
    }

    /**
     * Find account info.
     * 
     * @param address account address
     * @param ksPassword key store password
     * @return account info
     * @throws BusinessException business exception
     */
    public String queryUserInfo(String address, String ksPassword)
        throws BusinessException {
        return accountService.findInfo(address, ksPassword);
    }

    /**
     * Query account history.
     * 
     * @param address account address
     * @return list of accounts
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryAccountHistory(String address)
        throws BusinessException {
        return accountService.queryHistory(address);
    }

    /**
     * Find latest Account.
     * 
     * @param address account address
     * @return account data
     * @throws BusinessException business exception
     */
    public JSONObject findLatestAccount(String address)
        throws BusinessException {
        return accountService.findLatestWithoutKs(address);
    }

    /**
     * Query select account's contract name list(up to 1000 records).
     * 
     * @param address account address
     * @return contract name list
     * @throws BusinessException business exception
     */
    public List<String> queryContractNamesByOwner(String address)
        throws BusinessException {
        return accountService.queryContractNames(address);
    }

    /**
     * Query select account's template name list(up to 1000 records).
     * 
     * @param address account address
     * @return template name list
     * @throws BusinessException business exception
     */
    public List<String> queryTemplateNamesByOwner(String address)
        throws BusinessException {
        return accountService.queryTemplateNames(address);
    }

    /**
     * Query account ledger(up to 1000 records).
     * 
     * @param address account address
     * @return ledger
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryLedger(String address)
        throws BusinessException {
        return accountService.queryLedger(address);
    }

    /**
     * Query account ledger history in selected contract.
     * 
     * @param address account address
     * @param contractName selected contract
     * @return ledger history
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryLedgerHistory(String address,
                                               String contractName)
        throws BusinessException {
        return accountService.queryLedgerHistory(address, contractName);
    }

    /**
     * Query account ledger history in selected contract with extra info.
     * 
     * @param address account address
     * @param contractName selected contract
     * @return ledger history
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryLedgerHistoryEx(String address,
                                                 String contractName)
        throws BusinessException {
        return accountService.queryLedgerHistoryEx(address, contractName);
    }

}
