package cn.tdchain.api.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.Trans;
import cn.tdchain.api.rpc.CeConnection;
import cn.tdchain.api.rpc.CeConnectionUtils;
import cn.tdchain.api.service.AccountService;
import cn.tdchain.api.service.ContractService;
import cn.tdchain.cb.constant.ContractKey;
import cn.tdchain.cb.constant.KeyAndType;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.domain.Contract;
import cn.tdchain.cb.domain.ContractState;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.util.Base64Util;
import cn.tdchain.cb.util.CollectionUtils;
import cn.tdchain.cb.util.JsonUtils;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.jbcc.Result;

/**
 * Service Implementation for Cipher.
 *
 * @version 1.0
 * @author bingoer.H 2018-12-04
 */
public class ContractServiceImpl implements ContractService {

    private AccountService accountService = new AccountServiceImpl();

    private CeConnection con = CeConnectionUtils.getConnection();

    @Override
    public JSONObject findByHash(String hash) throws BusinessException {
        if (StringUtils.isBlank(hash)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        Result<Trans> tx = con.getTransByHash(hash);
        if (tx == null) {
            throw new BusinessException(ResultConstants.HASH_NOT_EXISTS);
        }
        if (!tx.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, tx.getMsg()));
        }
        if (tx.getEntity() == null
                || StringUtils.isBlank(tx.getEntity().getData())) {
            throw new BusinessException(ResultConstants.NOT_CONTRACT);
        }
        if (tx.getEntity().getType().equals(KeyAndType.C)) {
            Contract contract = parseContract(tx.getEntity());
            if (contract == null) {
                throw new BusinessException(ResultConstants.NOT_CONTRACT);
            } else {
                String csHash = contract.getReferenceHash()
                        + tx.getEntity().getHeight().toString();
                JSONObject temp = JSONObject.parseObject(contract.toString());
                temp.put("csHash", csHash);
                return temp;
            }
        } else {
            throw new BusinessException(ResultConstants.NOT_CONTRACT);
        }
    }

    @Override
    public ContractState findStateByHash(String hash) throws BusinessException {
        if (StringUtils.isBlank(hash)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        Result<Trans> tx = con.getTransByHash(hash);
        if (tx == null) {
            throw new BusinessException(ResultConstants.HASH_NOT_EXISTS);
        }
        if (!tx.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, tx.getMsg()));
        }
        if (tx.getEntity() == null
                || StringUtils.isBlank(tx.getEntity().getData())) {
            throw new BusinessException(ResultConstants.NOT_CONTRACT_STATE);
        }
        if (tx.getEntity().getType().equals(KeyAndType.CS)) {
            ContractState cs = parseContractState(tx.getEntity());
            if (cs == null) {
                throw new BusinessException(ResultConstants.NOT_CONTRACT_STATE);
            } else {
                cs.setCompiled(null);
                return cs;
            }
        } else {
            throw new BusinessException(ResultConstants.NOT_CONTRACT_STATE);
        }
    }

    @Override
    public JSONObject findExByHash(String hash) throws BusinessException {
        if (StringUtils.isBlank(hash)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        Result<Trans> tx = con.getTransByHash(hash);
        if (tx == null) {
            throw new BusinessException(ResultConstants.HASH_NOT_EXISTS);
        }
        if (!tx.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, tx.getMsg()));
        }
        if (tx.getEntity() == null
                || StringUtils.isBlank(tx.getEntity().getData())) {
            throw new BusinessException(ResultConstants.NOT_CONTRACT);
        }
        if (tx.getEntity().getType().equals(KeyAndType.C)) {
            Contract contract = parseContract(tx.getEntity());
            if (contract == null) {
                throw new BusinessException(ResultConstants.NOT_CONTRACT);
            } else {
                String csHash = contract.getReferenceHash()
                        + tx.getEntity().getHeight().toString();
                ContractState cs = this.findStateByHash(csHash);
                JSONObject temp = JSONObject.parseObject(cs.toString());
                temp.put("createTime", contract.getCreateTime());
                temp.put("description", contract.getDescription());
                temp.put("timestamp", contract.getTimestamp());
                temp.put("version", contract.getVersion());
                temp.put("hash", tx.getEntity().getHash());
                temp.put("csHash", csHash);
                return temp;
            }
        } else {
            throw new BusinessException(ResultConstants.NOT_CONTRACT);
        }
    }

    private Contract parseContract(Trans tx) {
        if (tx == null || StringUtils.isBlank(tx.getData())) {
            return null;
        }
        Contract contract = JsonUtils.fromJson(tx.getData(), Contract.class);
        if (contract == null) {
            return null;
        }
        contract.setArgs(Base64Util.decoder(contract.getArgs()));
        return contract;
    }

    private ContractState parseContractState(Trans tx) {
        if (tx == null || StringUtils.isBlank(tx.getData())) {
            return null;
        }
        ContractState contract = JsonUtils.fromJson(tx.getData(),
                ContractState.class);
        if (contract == null) {
            return null;
        }
        parseOpr(contract.getOprMap(), ContractKey.CREATE_ARGS);
        parseOpr(contract.getOprMap(), ContractKey.UPDATE_ARGS);
        parseOpr(contract.getOprMap(), ContractKey.UPDATE_SYNC_ARGS);
        parseOpr(contract.getOprMap(), ContractKey.RUN_ARGS);
        return contract;
    }

    private void parseOpr(Map<String, Object> oprMap, String key) {
        if (oprMap.containsKey(key)) {
            oprMap.put(key, Base64Util.decoder((String) oprMap.get(key)));
        }
    }

    @Override
    public JSONObject findByName(String contractName) throws BusinessException {
        if (StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        String contractKey = KeyAndType.getContractKey(contractName);
        Result<Trans> tx = con.getNewTransByKey(contractKey);
        if (tx == null) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        if (!tx.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, tx.getMsg()));
        }
        if (tx.getEntity() == null
                || StringUtils.isBlank(tx.getEntity().getData())) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        Contract contract = parseContract(tx.getEntity());
        if (contract == null) {
            throw new BusinessException(ResultConstants.NOT_CONTRACT);
        } else {
            JSONObject temp = JSONObject.parseObject(contract.toString());
            temp.put("hash", tx.getEntity().getHash());
            temp.put("csHash", temp.get("referenceHash")
                    + tx.getEntity().getHeight().toString());
            return temp;
        }
    }

    @Override
    public List<JSONObject> queryByOwner(String address)
        throws BusinessException {
        List<JSONObject> contracts = new ArrayList<JSONObject>();

        if (StringUtils.isBlank(address)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        List<String> contractNameList = accountService
                .queryContractNames(address);
        if (CollectionUtils.isEmpty(contractNameList)) {
            return contracts;
        }
        for (String contractName : contractNameList) {
            Result<Trans> tx = con
                    .getNewTransByKey(KeyAndType.getContractKey(contractName));
            if (tx == null || !tx.isSuccess() || tx.getEntity() == null
                    || StringUtils.isBlank(tx.getEntity().getData())) {
                continue;
            }
            Contract contract = parseContract(tx.getEntity());
            if (contract != null) {
                JSONObject temp = JSONObject.parseObject(contract.toString());
                temp.put("hash", tx.getEntity().getHash());
                contracts.add(temp);
            }
        }
        return contracts;
    }

    @Override
    public List<JSONObject> queryContractHistory(String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        List<JSONObject> contracts = new ArrayList<JSONObject>();

        Result<List<Trans>> contractTxs = con
                .getTransHistoryByKey(KeyAndType.getContractKey(contractName));
        if (contractTxs == null) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        if (!contractTxs.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, contractTxs.getMsg()));
        }
        if (CollectionUtils.isEmpty(contractTxs.getEntity())) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        contractTxs.getEntity().forEach(tx -> {
            Contract contract = parseContract(tx);
            if (contract != null) {
                JSONObject temp = JSONObject.parseObject(contract.toString());
                temp.put("hash", tx.getHash());
                temp.put("csHash",
                        temp.get("referenceHash") + tx.getHeight().toString());
                contracts.add(temp);
            }
        });
        return contracts;
    }

    @Override
    public List<JSONObject> queryContractStateHistory(String contractName)
        throws BusinessException {
        List<JSONObject> contracts = new ArrayList<JSONObject>();

        if (StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }

        Result<List<Trans>> contractTxs = con.getTransHistoryByKey(
                KeyAndType.getContractStateKey(contractName));
        if (contractTxs == null) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        if (!contractTxs.isSuccess()) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.GETCHAIN_FAILED, contractTxs.getMsg()));
        }
        if (CollectionUtils.isEmpty(contractTxs.getEntity())) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }

        contractTxs.getEntity().forEach(tx -> {
            ContractState cs = parseContractState(tx);
            if (cs != null) {
                JSONObject temp = JSONObject.parseObject(cs.toString());
                temp.remove("compiled");
                temp.put("hash", tx.getHash());
                contracts.add(temp);
            }
        });
        return contracts;

    }

    @Override
    public String queryLatestCsHash(String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        Result<Trans> latest = con
                .getNewTransByKey(KeyAndType.getContractStateKey(contractName));
        if (latest == null || !latest.isSuccess() || latest.getEntity() == null
                || StringUtils.isBlank(latest.getEntity().getData())) {
            return null;
        }
        return latest.getEntity().getHash();
    }

}
