package cn.tdchain.api;

import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.api.rpc.CeConnection;
import cn.tdchain.api.rpc.CeConnectionUtils;
import cn.tdchain.api.service.TemplateService;
import cn.tdchain.api.service.impl.TemplateServiceImpl;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.domain.Template;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.jbcc.rpc.RPCMessage;

/**
 * Template API.
 * 
 * @version 1.0
 * @author Bingoer.H 2018-12-05
 */
public class TemplateApi {

    private TemplateService templateService = new TemplateServiceImpl();
    private CeConnection con = CeConnectionUtils.getConnection();

    /**
     * Create contract template.
     * 
     * @param name contract template name
     * @param description template description
     * @param address creator address
     * @param ksPassword key store password
     * @param javaClassName java class name e.g. "cn.tdchain.Contract"
     * @param templateSourceCode java source code
     * @return template hash
     * @throws BusinessException business exception
     */
    public Map<String, String> create(String name, String description,
                                      String address, String ksPassword,
                                      String javaClassName,
                                      String templateSourceCode)
        throws BusinessException {
        if (StringUtils.isBlank(name) || StringUtils.isBlank(address)
                || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(javaClassName)
                || StringUtils.isBlank(templateSourceCode)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (name.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, name));
        }
        if (address.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, address));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("name", name);
        msg.getCommand().put("description", description);
        msg.getCommand().put("address", address);
        msg.getCommand().put("ksPassword", ksPassword);
        msg.getCommand().put("javaClassName", javaClassName);
        msg.getCommand().put("templateSourceCode", templateSourceCode);
        msg.setTargetType(RPCMessage.TargetType.CREATE_TEMPLATE);
        return con.sendAndReturn(msg, ResultConstants.TEMPLATE_CREATE_FAILED);
    }

    /**
     * Query template by transaction hash.
     * 
     * @param txHash transaction hash
     * @return template data
     * @throws BusinessException business exception
     */
    public Template findByHash(String txHash) throws BusinessException {
        return templateService.findByHash(txHash);
    }

    /**
     * Query template by name.
     * 
     * @param name template name
     * @return template data
     * @throws BusinessException business exception
     */
    public JSONObject findByName(String name) throws BusinessException {
        return templateService.findByName(name);
    }

}
