package cn.tdchain.api.service;

import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.cb.exception.BusinessException;

/**
 * Service for account.
 * 
 * @version 1.0
 * @author Bingoer.H 2018-11-29
 */
public interface AccountService {

    /**
     * Create super account.
     * 
     * @return super account address and hash
     * @throws BusinessException business exception
     */
    Map<String, String> createSuper() throws BusinessException;

    /**
     * Create account.
     * 
     * @param ksPassword key store password
     * @param crypto RSA or SM
     * @param info 用户个人信息 Map对象的json序列化字符串
     * @return account address and hash
     * @throws BusinessException business exception
     */
    Map<String, String> create(String ksPassword, String crypto, String info)
        throws BusinessException;

    /**
     * Find latest account without key store info.
     * 
     * @param address account address
     * @return account data without key store info
     * @throws BusinessException business exception
     */
    JSONObject findLatestWithoutKs(String address) throws BusinessException;

    /**
     * Find account info.
     * 
     * @param address account address
     * @param ksPassword key store password
     * @return account info
     * @throws BusinessException business exception
     */
    String findInfo(String address, String ksPassword) throws BusinessException;

    /**
     * Query account history.
     * 
     * @param address account address
     * @return list of accounts
     * @throws BusinessException business exception
     */
    List<JSONObject> queryHistory(String address) throws BusinessException;

    /**
     * Query select account's contract name list.
     * 
     * @param address account address
     * @return contract name list
     * @throws BusinessException business exception
     */
    List<String> queryContractNames(String address) throws BusinessException;

    /**
     * Query select account's template name list(up to 1000 records).
     * 
     * @param address account address
     * @return template name list
     * @throws BusinessException business exception
     */
    List<String> queryTemplateNames(String address) throws BusinessException;

    /**
     * Query account ledger(up to 1000 records).
     * 
     * @param address account address
     * @return ledger
     * @throws BusinessException business exception
     */
    List<JSONObject> queryLedger(String address) throws BusinessException;

    /**
     * Query account ledger history in selected contract.
     * 
     * @param address account address
     * @param contractName selected contract
     * @return ledger history
     * @throws BusinessException business exception
     */
    List<JSONObject> queryLedgerHistory(String address, String contractName)
        throws BusinessException;

    /**
     * Query account ledger history in selected contract with extra info.
     * 
     * @param address account address
     * @param contractName selected contract
     * @return ledger history
     * @throws BusinessException business exception
     */
    List<JSONObject> queryLedgerHistoryEx(String address, String contractName)
        throws BusinessException;

    /**
     * Encrypt user info.
     * 
     * @param address account address
     * @param ksPassword key store password
     * @param userInfo String
     * @return encrypt info
     * @throws BusinessException business exception
     */
    String encryptInfo(String address, String ksPassword, String userInfo)
        throws BusinessException;

    /**
     * Get account assets, templates, contracts counts.
     * 
     * @param address account address
     * @return JSONObject, key: "asset", "template", "contract"
     * @throws BusinessException business exception
     */
    JSONObject getAccountStatistic(String address) throws BusinessException;

}
